<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="2.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:fo="http://www.w3.org/1999/XSL/Format"
  xmlns:xlink="http://www.w3.org/1999/xlink"
  xmlns:mml="http://www.w3.org/1998/Math/MathML"
  xmlns:axf="http://www.antennahouse.com/names/XSL/Extensions"
  exclude-result-prefixes="xlink mml">

  <!-- J-Stage3用スタイルシート -->
  <!-- by Antenna House, Inc., 2012-03-30, 2014-07-07改訂 -->

  <!-- JATS Preview用スタイルシートをインポート -->
  <xsl:import href="jats-xslfo.xsl"/>

  <!--
    NOTE:
    
    本スタイルシートで定義しているトップレベルの parameter, variable, 
    attribute-set, template のうち、インポートされた JATS Preview 用
    スタイルシート jats-xslfo.xsl で定義されているものを再定義するもの
    には、コメントに [jats-xslfo] と記しています。
    
    同様に、スタイルシート xhtml-tables-fo.xsl で定義されているものを
    再定義するものには、コメントに [xhtml-tables-fo] と記しています。
    
    attribute-set の場合はインポートされた定義に追加することになるため、
    定義内容を知るには元の定義も参照する必要があります。
  -->

  <!-- ============================================================= -->
  <!-- TOP-LEVEL PARAMETERS                                          -->
  <!-- ============================================================= -->

  <!-- 表紙を出力するか -->
  <xsl:param name="output-cover-page" select="false()"/>

  <!-- 文書の診断結果を出力するか -->
  <xsl:param name="output-diagnostics" select="false()"/>

  <!-- ============================================================= -->
  <!-- TYPOGRAPHIC SPECIFICATIONS                                    -->
  <!-- ============================================================= -->

  <!-- 見出し用フォント [jats-xslfo] -->
  <xsl:variable name="titlefont">sans-serif</xsl:variable>

  <!-- 本文用フォント [jats-xslfo] -->
  <xsl:variable name="textfont">serif</xsl:variable>

  <!-- 本文の文字サイズ [jats-xslfo] -->
  <xsl:variable name="textsize">9pt</xsl:variable>

  <!-- 本文の行送り [jats-xslfo] -->
  <xsl:variable name="textleading">14pt</xsl:variable>

  <!-- 本文の左インデント [jats-xslfo] -->
  <xsl:variable name="mainindent">0pt</xsl:variable>

  <!-- ページの大きさと余白 -->
  <xsl:variable name="page-width">210mm</xsl:variable>
  <xsl:variable name="page-height">278mm</xsl:variable>
  <xsl:variable name="page-margin-inside">21mm</xsl:variable>
  <xsl:variable name="page-margin-outside">21mm</xsl:variable>
  <xsl:variable name="page-margin-top">10mm</xsl:variable>
  <xsl:variable name="page-margin-bottom">13mm</xsl:variable>
  <xsl:variable name="page-body-margin-top">8mm</xsl:variable>
  <xsl:variable name="page-body-margin-bottom">0mm</xsl:variable>
  <xsl:variable name="page-column-count">2</xsl:variable>
  <xsl:variable name="page-column-gap">9mm</xsl:variable>

  <!-- ============================================================= -->
  <!-- Attribute sets                                                -->
  <!-- ============================================================= -->

  <!-- 基本となる設定（fo:root に指定されるもの） -->
  <xsl:attribute-set name="root">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$textfont"/>
    </xsl:attribute>
    <xsl:attribute name="font-size">
      <xsl:value-of select="$textsize"/>
    </xsl:attribute>
    <xsl:attribute name="line-height">
      <xsl:value-of select="$textleading"/>
    </xsl:attribute>
    <xsl:attribute name="widows">1</xsl:attribute>
    <xsl:attribute name="orphans">1</xsl:attribute>
    <xsl:attribute name="line-height-shift-adjustment"
      >disregard-shifts</xsl:attribute>
    <xsl:attribute name="line-stacking-strategy">font-height</xsl:attribute>
    <xsl:attribute name="text-align">justify</xsl:attribute>
    <!-- 以下AH拡張 -->
    <xsl:attribute name="axf:ligature-mode">latin</xsl:attribute>
    <xsl:attribute name="axf:kerning-mode">pair</xsl:attribute>
    <xsl:attribute name="axf:hanging-punctuation">none</xsl:attribute>
    <xsl:attribute name="axf:punctuation-trim">start adjacent
      end</xsl:attribute>
    <xsl:attribute name="axf:text-justify-trim">punctuation kana</xsl:attribute>
    <xsl:attribute name="axf:avoid-widow-words">true</xsl:attribute>
    <xsl:attribute name="axf:text-autospace">ideograph-numeric
      ideograph-alpha</xsl:attribute>
    <xsl:attribute name="axf:text-autospace-width">25%</xsl:attribute>
  </xsl:attribute-set>

  <!-- 本文 [jats-xslfo] -->
  <xsl:attribute-set name="body">
    <xsl:attribute name="text-align">justify</xsl:attribute>
  </xsl:attribute-set>

  <!-- ページヘッダー [jats-xslfo] -->
  <xsl:attribute-set name="page-header">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$textfont"/>
    </xsl:attribute>
    <xsl:attribute name="font-size">8pt</xsl:attribute>
    <xsl:attribute name="line-height">8pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- ノンブル -->
  <xsl:attribute-set name="page-header-pageno">
    <xsl:attribute name="font-family">serif</xsl:attribute>
    <xsl:attribute name="font-weight">bold</xsl:attribute>
    <xsl:attribute name="font-size">10pt</xsl:attribute>
    <xsl:attribute name="start-indent">9pt</xsl:attribute>
    <xsl:attribute name="end-indent">9pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 見出し [jats-xslfo] -->
  <xsl:attribute-set name="title">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$titlefont"/>
    </xsl:attribute>
    <xsl:attribute name="font-weight">bold</xsl:attribute>
    <xsl:attribute name="keep-with-next.within-column">always</xsl:attribute>
  </xsl:attribute-set>

  <!-- 大見出し [jats-xslfo] -->
  <xsl:attribute-set name="main-title" use-attribute-sets="title">
    <xsl:attribute name="font-size">10.5pt</xsl:attribute>
    <xsl:attribute name="line-height">
      <xsl:value-of select="$textleading"/>
    </xsl:attribute>
    <xsl:attribute name="space-after">
      <xsl:value-of select="$textleading"/>
    </xsl:attribute>
  </xsl:attribute-set>

  <!-- 中見出し [jats-xslfo] -->
  <xsl:attribute-set name="section-title" use-attribute-sets="title">
    <xsl:attribute name="font-size">9pt</xsl:attribute>
    <xsl:attribute name="line-height">
      <xsl:value-of select="$textleading"/>
    </xsl:attribute>
  </xsl:attribute-set>

  <!-- 小見出し [jats-xslfo] -->
  <xsl:attribute-set name="subsection-title" use-attribute-sets="title">
    <xsl:attribute name="font-size">9pt</xsl:attribute>
    <xsl:attribute name="font-weight">normal</xsl:attribute>
    <xsl:attribute name="line-height">
      <xsl:value-of select="$textleading"/>
    </xsl:attribute>
  </xsl:attribute-set>

  <!-- セクション見出しのラベル -->
  <xsl:attribute-set name="section-label">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$textfont"/>
    </xsl:attribute>
  </xsl:attribute-set>

  <!-- 図表キャプションのラベル -->
  <xsl:attribute-set name="caption-label">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$titlefont"/>
    </xsl:attribute>
    <xsl:attribute name="font-size">8pt</xsl:attribute>
    <xsl:attribute name="font-weight">normal</xsl:attribute>
  </xsl:attribute-set>

  <!-- 記事カテゴリ -->
  <xsl:attribute-set name="subject" use-attribute-sets="title">
    <xsl:attribute name="font-size">10.5pt</xsl:attribute>
    <xsl:attribute name="line-height">14pt</xsl:attribute>
    <xsl:attribute name="letter-spacing">0.3em</xsl:attribute>
    <xsl:attribute name="space-before">4pt</xsl:attribute>
    <xsl:attribute name="space-before.conditionality">retain</xsl:attribute>
  </xsl:attribute-set>

  <!-- 記事タイトル [jats-xslfo] -->
  <xsl:attribute-set name="firstpage-title" use-attribute-sets="title">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$textfont"/>
    </xsl:attribute>
    <xsl:attribute name="font-weight">normal</xsl:attribute>
    <xsl:attribute name="font-size">13pt</xsl:attribute>
    <xsl:attribute name="line-height">16pt</xsl:attribute>
    <xsl:attribute name="space-before">24pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 翻訳タイトル [jats-xslfo] -->
  <xsl:attribute-set name="firstpage-trans-title" use-attribute-sets="title">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$textfont"/>
    </xsl:attribute>
    <xsl:attribute name="font-weight">normal</xsl:attribute>
    <xsl:attribute name="font-size">13pt</xsl:attribute>
    <xsl:attribute name="line-height">16pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 著者情報ブロック [jats-xslfo] -->
  <xsl:attribute-set name="contrib-block">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$textfont"/>
    </xsl:attribute>
    <xsl:attribute name="font-weight">normal</xsl:attribute>
    <xsl:attribute name="font-size">8pt</xsl:attribute>
    <xsl:attribute name="line-height">11pt</xsl:attribute>
    <xsl:attribute name="space-before">24pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 翻訳版著者情報ブロック -->
  <xsl:attribute-set name="trans-contrib-block">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$textfont"/>
    </xsl:attribute>
    <xsl:attribute name="font-weight">normal</xsl:attribute>
    <xsl:attribute name="font-size">9pt</xsl:attribute>
    <xsl:attribute name="line-height">12pt</xsl:attribute>
    <xsl:attribute name="space-before">18pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 著者名 -->
  <xsl:attribute-set name="authors">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$textfont"/>
    </xsl:attribute>
    <xsl:attribute name="font-weight">normal</xsl:attribute>
    <xsl:attribute name="font-size">10.5pt</xsl:attribute>
    <xsl:attribute name="line-height">16pt</xsl:attribute>
    <xsl:attribute name="space-after">16pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 翻訳版著者名 -->
  <xsl:attribute-set name="trans-authors">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$textfont"/>
    </xsl:attribute>
    <xsl:attribute name="font-weight">normal</xsl:attribute>
    <xsl:attribute name="font-size">9pt</xsl:attribute>
    <xsl:attribute name="line-height">12pt</xsl:attribute>
    <xsl:attribute name="space-after">12pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 抄録 [jats-xslfo] -->
  <xsl:attribute-set name="abstract">
    <xsl:attribute name="start-indent">28mm</xsl:attribute>
    <xsl:attribute name="font-size">9pt</xsl:attribute>
    <xsl:attribute name="line-height">14pt</xsl:attribute>
    <xsl:attribute name="space-before">12pt</xsl:attribute>
    <xsl:attribute name="space-after">34pt</xsl:attribute>
    <xsl:attribute name="space-after.conditionality">retain</xsl:attribute>
  </xsl:attribute-set>

  <!-- 翻訳抄録 -->
  <xsl:attribute-set name="trans-abstract">
    <xsl:attribute name="start-indent">28mm</xsl:attribute>
    <xsl:attribute name="font-size">9pt</xsl:attribute>
    <xsl:attribute name="line-height">12pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- キーワードリスト -->
  <xsl:attribute-set name="keywords">
    <xsl:attribute name="space-before">14pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 翻訳版キーワードリスト -->
  <xsl:attribute-set name="trans-keywords">
    <xsl:attribute name="space-before">12pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 節 [jats-xslfo] -->
  <xsl:attribute-set name="section">
    <xsl:attribute name="space-before">
      <xsl:value-of select="$textleading"/>
    </xsl:attribute>
  </xsl:attribute-set>

  <!-- 段落 [jats-xslfo] -->
  <xsl:attribute-set name="paragraph">
    <xsl:attribute name="space-before">0pt</xsl:attribute>
    <xsl:attribute name="text-indent">1em</xsl:attribute>
  </xsl:attribute-set>

  <!-- メールアドレス [jats-xslfo] -->
  <xsl:attribute-set name="email">
    <xsl:attribute name="font-family">
      <xsl:value-of select="$textfont"/>
    </xsl:attribute>
  </xsl:attribute-set>

  <!-- 図版やテーブルの囲み [jats-xslfo] -->
  <xsl:attribute-set name="box">
    <!-- 罫線囲みは無し -->
    <xsl:attribute name="border"/>
    <xsl:attribute name="padding">0pt</xsl:attribute>
    <xsl:attribute name="space-before">8pt</xsl:attribute>
    <xsl:attribute name="space-after">8pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- フロート配置 [jats-xslfo] -->
  <xsl:attribute-set name="float">
    <!-- AH拡張float -->
    <xsl:attribute name="axf:float">column top</xsl:attribute>
    <xsl:attribute name="axf:float-margin-y">10pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- テーブルの囲み [jats-xslfo] -->
  <xsl:attribute-set name="table-box" use-attribute-sets="box">
    <xsl:attribute name="font-size">8pt</xsl:attribute>
    <xsl:attribute name="line-height">12pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 図版の囲み [jats-xslfo] -->
  <xsl:attribute-set name="fig-box" use-attribute-sets="box">
    <xsl:attribute name="font-size">8pt</xsl:attribute>
    <xsl:attribute name="line-height">12pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 画像の配置 [jats-xslfo] -->
  <xsl:attribute-set name="media-object">
    <xsl:attribute name="text-align">center</xsl:attribute>
    <xsl:attribute name="space-after">3pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 図のキャプション -->
  <xsl:attribute-set name="fig-caption" use-attribute-sets="caption">
    <xsl:attribute name="start-indent">27pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- インデントをリセット -->
  <xsl:attribute-set name="reset-indents">
    <xsl:attribute name="start-indent">0pt</xsl:attribute>
    <xsl:attribute name="end-indent">0pt</xsl:attribute>
    <xsl:attribute name="text-indent">0pt</xsl:attribute>
    <xsl:attribute name="last-line-end-indent">0pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- table [xhtml-tables-fo] -->
  <xsl:attribute-set name="table">
    <xsl:attribute name="border-collapse">separate</xsl:attribute>
    <xsl:attribute name="border-spacing">0pt</xsl:attribute>
    <xsl:attribute name="border"/>
    <xsl:attribute name="border-top">solid 0.25pt</xsl:attribute>
    <xsl:attribute name="border-bottom">solid 0.25pt</xsl:attribute>
    <xsl:attribute name="padding-top">2pt</xsl:attribute>
    <xsl:attribute name="space-before">5pt</xsl:attribute>
    <xsl:attribute name="space-after">5pt</xsl:attribute>
    <xsl:attribute name="width">100%</xsl:attribute>
  </xsl:attribute-set>

  <!-- table header cell [xhtml-tables-fo] -->
  <xsl:attribute-set name="th">
    <xsl:attribute name="font-weight">normal</xsl:attribute>
    <xsl:attribute name="border"/>
    <xsl:attribute name="border-top">solid 0.25pt</xsl:attribute>
    <xsl:attribute name="border-bottom">solid 0.25pt</xsl:attribute>
    <xsl:attribute name="padding">2.5pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- table data cell [xhtml-tables-fo] -->
  <xsl:attribute-set name="td">
    <xsl:attribute name="border"/>
    <xsl:attribute name="padding">0pt 2.5pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 脚注配置 -->
  <xsl:attribute-set name="footnote-position">
    <!-- 脚注を段に配置（AH拡張） -->
    <xsl:attribute name="axf:footnote-position">column</xsl:attribute>
  </xsl:attribute-set>

  <!-- 脚注配置 [jats-xslfo] -->
  <xsl:attribute-set name="footnote-body" use-attribute-sets="reset-indents">
    <xsl:attribute name="space-before">0pt</xsl:attribute>
    <xsl:attribute name="font-size">8pt</xsl:attribute>
    <xsl:attribute name="line-height">10pt</xsl:attribute>
  </xsl:attribute-set>

  <!-- 脚注本体 [jats-xslfo] -->
  <xsl:attribute-set name="endnote">
    <xsl:attribute name="start-indent">1em</xsl:attribute>
  </xsl:attribute-set>

  <!-- 引用文献 [jats-xslfo] -->
  <xsl:attribute-set name="citation" use-attribute-sets="paragraph">
    <xsl:attribute name="font-size">8pt</xsl:attribute>
    <xsl:attribute name="line-height">12pt</xsl:attribute>
    <xsl:attribute name="start-indent">2em</xsl:attribute>
    <xsl:attribute name="text-indent">-2em</xsl:attribute>
  </xsl:attribute-set>

  <!-- ============================================================= -->
  <!-- TOP-LEVEL TEMPLATES                                           -->
  <!-- ============================================================= -->


  <xsl:template match="/">
    <fo:root xml:lang="{/article/@xml:lang}" xsl:use-attribute-sets="root">
      <fo:layout-master-set>
        <xsl:call-template name="define-simple-page-masters"/>
        <xsl:call-template name="define-page-sequences"/>
      </fo:layout-master-set>
      <xsl:apply-templates/>
    </fo:root>
  </xsl:template>


  <xsl:template match="article">
    <xsl:if test="$output-cover-page">
      <!-- このパラメータがtrueのときだけカバーページ出力 -->
      <fo:page-sequence master-reference="cover-sequence"
        force-page-count="even">
        <xsl:call-template name="define-footnote-separator"/>
        <fo:flow flow-name="body">
          <fo:block line-stacking-strategy="font-height"
            line-height-shift-adjustment="disregard-shifts">
            <xsl:call-template name="set-article-cover-page"/>
          </fo:block>
        </fo:flow>
      </fo:page-sequence>
    </xsl:if>

    <!-- Populate the content sequence -->
    <fo:page-sequence master-reference="content-sequence"
      initial-page-number="1">
      <!-- ページ番号初期値 -->
      <xsl:if test="/article/front/article-meta/fpage">
        <xsl:attribute name="initial-page-number"
          select="/article/front/article-meta/fpage"/>
      </xsl:if>

      <!-- 記事の先頭ページのページヘッダー -->
      <fo:static-content flow-name="first-header">
        <fo:block xsl:use-attribute-sets="page-header">
          <xsl:call-template name="make-page-header">
            <xsl:with-param name="face" select="'first'"/>
          </xsl:call-template>
        </fo:block>
        <xsl:call-template name="first-header-rule"/>
      </fo:static-content>

      <fo:static-content flow-name="recto-header">
        <fo:block xsl:use-attribute-sets="page-header">
          <xsl:call-template name="make-page-header">
            <xsl:with-param name="face" select="'recto'"/>
          </xsl:call-template>
        </fo:block>
      </fo:static-content>
      <fo:static-content flow-name="verso-header">
        <fo:block xsl:use-attribute-sets="page-header">
          <xsl:call-template name="make-page-header">
            <xsl:with-param name="face" select="'verso'"/>
          </xsl:call-template>
        </fo:block>
      </fo:static-content>
      <xsl:call-template name="define-footnote-separator"/>
      <fo:flow flow-name="body">
        <fo:block>

          <!-- set the article opener, body, and backmatter -->

          <!-- 前付け -->
          <xsl:call-template name="set-article-opener"/>

          <!-- 連絡先情報など -->
          <xsl:call-template name="set-correspondence-note"/>

          <!-- 記事本体 -->
          <xsl:call-template name="set-article"/>

          <!-- 終わりに翻訳抄録ブロック -->
          <xsl:call-template name="trans-abstract-block"/>

        </fo:block>

      </fo:flow>
    </fo:page-sequence>

    <!-- produce document diagnostics after the end of 
       the article; this has a page sequence in it
       and all else needed -->
    <xsl:if test="$output-diagnostics">
      <!-- このパラメータがtrueのときだけ診断実行 -->
      <xsl:call-template name="run-diagnostics"/>
    </xsl:if>
  </xsl:template>

  <!-- ページマスター定義 [jats-xslfo] -->
  <xsl:template name="define-simple-page-masters">
    <!-- blank and cover are similar -->
    <!-- cover page uses recto margins -->
    <fo:simple-page-master master-name="cover" page-height="{$page-height}"
      page-width="{$page-width}" margin-top="{$page-margin-top}"
      margin-bottom="{$page-margin-bottom}" margin-left="{$page-margin-inside}"
      margin-right="{$page-margin-outside}">
      <fo:region-body region-name="body" margin-top="{$page-body-margin-top}"
        margin-bottom="{$page-body-margin-bottom}" margin-left="0pt"
        margin-right="0pt"/>
    </fo:simple-page-master>

    <!-- blank page -->
    <fo:simple-page-master master-name="blank" page-height="{$page-height}"
      page-width="{$page-width}" margin-top="{$page-margin-top}"
      margin-bottom="{$page-margin-bottom}" margin-left="{$page-margin-outside}"
      margin-right="{$page-margin-inside}">
      <fo:region-body region-name="body" margin-top="{$page-body-margin-top}"
        margin-bottom="{$page-body-margin-bottom}" margin-left="0pt"
        margin-right="0pt"/>
    </fo:simple-page-master>

    <!-- first has recto margins -->
    <fo:simple-page-master master-name="first" page-height="{$page-height}"
      page-width="{$page-width}" margin-top="{$page-margin-top}"
      margin-bottom="{$page-margin-bottom}" margin-left="{$page-margin-inside}"
      margin-right="{$page-margin-outside}">
      <fo:region-body region-name="body" margin-top="{$page-body-margin-top}"
        margin-bottom="{$page-body-margin-bottom}" margin-left="0pt"
        margin-right="0pt" column-count="{$page-column-count}"
        column-gap="{$page-column-gap}"
        xsl:use-attribute-sets="footnote-position"/>
      <fo:region-before region-name="first-header"
        extent="{$page-body-margin-top}"/>
    </fo:simple-page-master>

    <!-- verso page -->
    <fo:simple-page-master master-name="verso" page-height="{$page-height}"
      page-width="{$page-width}" margin-top="{$page-margin-top}"
      margin-bottom="{$page-margin-bottom}" margin-left="{$page-margin-outside}"
      margin-right="{$page-margin-inside}">
      <fo:region-body region-name="body" margin-top="{$page-body-margin-top}"
        margin-bottom="{$page-body-margin-bottom}" margin-left="0pt"
        margin-right="0pt" column-count="{$page-column-count}"
        column-gap="{$page-column-gap}"
        xsl:use-attribute-sets="footnote-position"/>
      <fo:region-before region-name="verso-header"
        extent="{$page-body-margin-top}"/>
    </fo:simple-page-master>

    <!-- recto page -->
    <fo:simple-page-master master-name="recto" page-height="{$page-height}"
      page-width="{$page-width}" margin-top="{$page-margin-top}"
      margin-bottom="{$page-margin-bottom}" margin-left="{$page-margin-inside}"
      margin-right="{$page-margin-outside}">
      <fo:region-body region-name="body" margin-top="{$page-body-margin-top}"
        margin-bottom="{$page-body-margin-bottom}" margin-left="0pt"
        margin-right="0pt" column-count="{$page-column-count}"
        column-gap="{$page-column-gap}"
        xsl:use-attribute-sets="footnote-position"/>
      <fo:region-before region-name="recto-header"
        extent="{$page-body-margin-top}"/>
    </fo:simple-page-master>
  </xsl:template>

  <!-- 脚注の区切り線 [jats-xslfo] -->
  <xsl:template name="define-footnote-separator">
    <fo:static-content flow-name="xsl-footnote-separator">
      <fo:block space-before="6pt" space-before.conditionality="retain"
        space-after="6pt" border-bottom="solid 0.25pt"/>
    </fo:static-content>
  </xsl:template>

  <!-- ページヘッダー作成 [jats-xslfo] -->
  <xsl:template name="make-page-header">
    <!-- Pass $face in as 'recto' or 'verso' to get titles and page nos
       on facing pages -->
    <xsl:param name="face"/>
    <xsl:param name="dummy-cell">
      <fo:block/>
    </xsl:param>
    <xsl:param name="center-cell">
      <fo:block/>
    </xsl:param>
    <fo:table border-style="none" width="100%">
      <fo:table-body>
        <fo:table-row>
          <xsl:choose>
            <xsl:when test="$face='first'">
              <fo:table-cell xsl:use-attribute-sets="page-header-pageno-cell">
                <xsl:copy-of select="$dummy-cell"/>
              </fo:table-cell>
            </xsl:when>
            <xsl:when test="$face='recto'">
              <fo:table-cell xsl:use-attribute-sets="page-header-pageno-cell">
                <xsl:copy-of select="$dummy-cell"/>
              </fo:table-cell>
            </xsl:when>
            <xsl:when test="$face='verso'">
              <fo:table-cell xsl:use-attribute-sets="page-header-pageno-cell">
                <fo:block text-align="left"
                  xsl:use-attribute-sets="page-header-pageno">
                  <fo:page-number/>
                </fo:block>
              </fo:table-cell>
            </xsl:when>
            <xsl:otherwise/>
          </xsl:choose>

          <xsl:choose>
            <xsl:when test="$face='first'">
              <fo:table-cell xsl:use-attribute-sets="page-header-pageno-cell">
                <xsl:copy-of select="$center-cell"/>
              </fo:table-cell>
            </xsl:when>
            <xsl:when test="$face='recto'">
              <fo:table-cell xsl:use-attribute-sets="page-header-title-cell">
                <fo:block text-align="center">
                  <xsl:call-template name="recto-page-header-title"/>
                </fo:block>
              </fo:table-cell>
            </xsl:when>
            <xsl:when test="$face='verso'">
              <fo:table-cell xsl:use-attribute-sets="page-header-title-cell">
                <fo:block text-align="center">
                  <xsl:call-template name="verso-page-header-title"/>
                </fo:block>
              </fo:table-cell>
            </xsl:when>
          </xsl:choose>

          <xsl:choose>
            <xsl:when test="$face='first'">
              <fo:table-cell xsl:use-attribute-sets="page-header-title-cell">
                <fo:block text-align="right">
                  <xsl:call-template name="first-page-header-title"/>
                </fo:block>
              </fo:table-cell>
            </xsl:when>
            <xsl:when test="$face='verso'">
              <fo:table-cell xsl:use-attribute-sets="page-header-pageno-cell">
                <xsl:copy-of select="$dummy-cell"/>
              </fo:table-cell>
            </xsl:when>
            <xsl:when test="$face='recto'">
              <fo:table-cell xsl:use-attribute-sets="page-header-pageno-cell">
                <fo:block text-align="right"
                  xsl:use-attribute-sets="page-header-pageno">
                  <fo:page-number/>
                </fo:block>
              </fo:table-cell>
            </xsl:when>
            <xsl:otherwise/>
          </xsl:choose>
        </fo:table-row>
      </fo:table-body>
    </fo:table>
  </xsl:template>

  <!-- 前付け区切り罫 [jats-xslfo] -->
  <xsl:template name="banner-rule">
    <fo:block space-before="8pt" space-after="8pt" border-bottom="solid 0.25pt"
    />
  </xsl:template>

  <!-- 先頭ヘッダー罫 -->
  <xsl:template name="first-header-rule">
    <!-- 子持ち罫 -->
    <fo:block space-before="8pt" space-after="8pt" border-top="solid 1pt"
      border-bottom="solid 0.25pt" padding-top="1.25pt"/>
  </xsl:template>


  <!-- 翻訳言語 -->
  <xsl:variable name="trans-lang">
    <xsl:value-of select="/article/front/article-meta/trans-abstract/@xml:lang"
    />
  </xsl:variable>

  <!-- 翻訳抄録のヘッダータイトル -->
  <xsl:template name="trans-header-title">
    <fo:inline>
      <xsl:value-of
        select="/article/front/journal-meta/journal-title-group[1]/journal-title[@xml:lang=$trans-lang]"/>
      <xsl:text>&#x2002;</xsl:text>
      <xsl:value-of select="/article/front/article-meta[1]/volume[1]"/>
      <xsl:text>: </xsl:text>
      <xsl:value-of select="/article/front/article-meta[1]/fpage[1]"/>
      <xsl:text>&#x2013;</xsl:text>
      <xsl:value-of select="/article/front/article-meta[1]/lpage[1]"/>
      <xsl:text> (</xsl:text>
      <xsl:value-of select="/article/front/article-meta[1]/pub-date[1]/year"/>
      <xsl:text>) </xsl:text>
    </fo:inline>
  </xsl:template>

  <!-- 以下、First-page、verso-page、及びrecto-pageの柱を個別に定義 -->
  <xsl:template name="first-page-header-title">
    <fo:inline>
      <xsl:value-of
        select="/article/front/journal-meta/journal-title-group[1]/journal-title[not(@xml:lang=$trans-lang)]"/>
      <xsl:text>&#x2002;</xsl:text>
      <xsl:value-of select="/article/front/article-meta[1]/volume[1]"/>
      <xsl:text>: </xsl:text>
      <xsl:value-of select="/article/front/article-meta[1]/fpage[1]"/>
      <xsl:text>&#x2013;</xsl:text>
      <xsl:value-of select="/article/front/article-meta[1]/lpage[1]"/>
      <xsl:text> (</xsl:text>
      <xsl:value-of select="/article/front/article-meta[1]/pub-date[1]/year"/>
      <xsl:text>) </xsl:text>
    </fo:inline>
  </xsl:template>

  <xsl:template name="verso-page-header-title">
    <xsl:for-each select="/article/front/article-meta/contrib-group/contrib">
      <fo:inline>
        <xsl:value-of
          select="./name-alternatives[1]/name[not(@xml:lang=$trans-lang)]/surname[1]"/>
        <xsl:if test="not(position()=last())">
          <xsl:text>・</xsl:text>
        </xsl:if>
      </fo:inline>
    </xsl:for-each>
  </xsl:template>

  <xsl:template name="recto-page-header-title">
    <fo:inline>
      <xsl:value-of
        select="/article/front/article-meta/title-group/article-title"/>
    </fo:inline>
  </xsl:template>


  <!-- 前付け [jats-xslfo] -->
  <xsl:template name="set-article-opener">
    <xsl:for-each select="/article/front/article-meta">
      <fo:block span="all">

        <!-- 記事カテゴリ -->
        <fo:block xsl:use-attribute-sets="subject">
          <xsl:apply-templates
            select="article-categories/subj-group[not(@xml:lang=$trans-lang)]/subject"
          />
        </fo:block>

        <!-- 記事タイトル部 -->
        <xsl:apply-templates select="title-group"/>

        <!-- 著者情報ブロック -->
        <fo:block xsl:use-attribute-sets="contrib-block">
          <!-- 著者名リスト -->
          <xsl:call-template name="authors"/>

          <!-- 所属リスト -->
          <xsl:apply-templates
            select="contrib-group//aff[not(@xml:lang=$trans-lang)]"
            mode="contrib"/>
        </fo:block>

        <!-- 抄録 -->
        <xsl:variable name="abstracts"
          select="abstract[not(@abstract-type='toc')]"/>

        <xsl:if test="$abstracts">
          <xsl:call-template name="banner-rule"/>
        </xsl:if>
        <xsl:apply-templates select="$abstracts"/>

      </fo:block>
    </xsl:for-each>
  </xsl:template>

  <!-- 記事タイトル部 -->
  <xsl:template match="title-group">
    <xsl:choose>
      <xsl:when test="/article/front/article-meta/trans-abstract">
        <!-- 翻訳抄録ありの場合は翻訳タイトルはここではなく翻訳抄録ブロック内に -->
        <xsl:apply-templates select="*[not(self::trans-title-group)]"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- 翻訳抄録ブロック -->
  <xsl:template name="trans-abstract-block">
    <xsl:for-each select="/article/front/article-meta[trans-abstract]">
      <fo:block span="all" space-before="17mm"
        space-before.conditionality="retain"
        xml:lang="{trans-abstract/@xml:lang}">

<fo:block keep-together.within-page="always">
        <!-- 翻訳抄録ページのヘッダー -->
        <fo:block xsl:use-attribute-sets="page-header">
          <xsl:call-template name="trans-header-title"/>
        </fo:block>

        <!-- 先頭ヘッダー罫 -->
        <xsl:call-template name="first-header-rule"/>

        <!-- 翻訳タイトル -->
        <xsl:apply-templates select="title-group/trans-title-group"/>

        <fo:block xsl:use-attribute-sets="trans-contrib-block">
          <!-- 翻訳版著者名リスト -->
          <xsl:call-template name="trans-authors"/>

          <!-- 翻訳版所属リスト -->
          <xsl:apply-templates
            select="contrib-group/aff-alternatives/aff[@xml:lang=$trans-lang]"
            mode="contrib"/>
        </fo:block>
</fo:block>

        <!-- 翻訳抄録 -->
        <xsl:variable name="trans-abstracts"
          select="trans-abstract[not(@abstract-type='toc')]"/>

        <xsl:if test="$trans-abstracts">
          <xsl:call-template name="banner-rule"/>
        </xsl:if>
        <xsl:apply-templates select="$trans-abstracts"/>
      </fo:block>
    </xsl:for-each>
  </xsl:template>

  <!-- 著者名リスト -->
  <xsl:template name="authors">
    <fo:block xsl:use-attribute-sets="authors">
      <xsl:for-each select="contrib-group/contrib">
        <xsl:apply-templates
          select=".//name[not(@xml:lang=$trans-lang)]/surname"/>
        <xsl:if
          test="string-length(.//name[not(@xml:lang=$trans-lang)]/surname)=1">
          <xsl:text>　</xsl:text>
        </xsl:if>
        <xsl:if
          test="string-length(.//name[not(@xml:lang=$trans-lang)]/given-names)=1">
          <xsl:text>　</xsl:text>
        </xsl:if>
        <xsl:apply-templates
          select=".//name[not(@xml:lang=$trans-lang)]/given-names"/>
        <fo:inline xsl:use-attribute-sets="superscript">
          <xsl:apply-templates select=".//xref[@ref-type='aff']"/>
        </fo:inline>
        <xsl:if test="not(position()=last())">
          <xsl:text>・</xsl:text>
        </xsl:if>
      </xsl:for-each>
    </fo:block>
  </xsl:template>

<!-- 著者の所属へのリンク番号 -->
<xsl:template match="xref[@ref-type='aff']">
	<xsl:value-of select="." />
	<xsl:choose>
		<xsl:when test="following-sibling::xref[@ref-type='aff']">
			<xsl:text>,</xsl:text>
		</xsl:when>
		<xsl:otherwise>
			<xsl:choose>
				<xsl:when test="substring(., string-length(.), 1) = ')'">
				</xsl:when>
				<xsl:otherwise>
					<!-- <xsl:text>)</xsl:text> -->
				</xsl:otherwise>
			</xsl:choose>
		</xsl:otherwise>
	</xsl:choose>
</xsl:template>

  <!-- 翻訳版著者名リスト -->
  <xsl:template name="trans-authors">
    <fo:block xsl:use-attribute-sets="trans-authors">
      <xsl:for-each select="contrib-group/contrib">
        <xsl:apply-templates select=".//name[@xml:lang=$trans-lang]/given-names"/>
        <xsl:text> </xsl:text>
        <xsl:apply-templates select=".//name[@xml:lang=$trans-lang]/surname"/>
        <fo:inline xsl:use-attribute-sets="superscript">
          <xsl:apply-templates select=".//xref[@ref-type='aff']"/>
        </fo:inline>
        <xsl:choose>
          <xsl:when test="position()&lt;last()-1">
            <xsl:text>, </xsl:text>
          </xsl:when>
          <xsl:when test="position()=last()-1">
            <xsl:text> and </xsl:text>
          </xsl:when>
        </xsl:choose>
      </xsl:for-each>
    </fo:block>
  </xsl:template>

  <!-- 抄録 -->
  <xsl:template match="abstract">
    <fo:block-container xsl:use-attribute-sets="abstract">
      <fo:block xsl:use-attribute-sets="reset-indents">

        <xsl:apply-templates/>
        <fo:block xsl:use-attribute-sets="keywords">
          <fo:block xsl:use-attribute-sets="title">
            <xsl:choose>
              <xsl:when test="/article/@xml:lang='ja'">
                <xsl:text>キーワード</xsl:text>
              </xsl:when>
              <xsl:otherwise>
                <xsl:text>Key Words</xsl:text>
              </xsl:otherwise>
            </xsl:choose>
          </fo:block>
          <!-- キーワードリスト -->
          <xsl:for-each select="../kwd-group[not(@xml:lang=$trans-lang)]/kwd">
            <fo:inline>
              <xsl:apply-templates/>
            </fo:inline>
            <xsl:if test="not(position()=last())">
              <xsl:choose>
                <xsl:when test="/article/@xml:lang='ja'">
                  <xsl:text>，</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:text>, </xsl:text>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:if>
          </xsl:for-each>
        </fo:block>
        <xsl:call-template name="banner-rule"/>
      </fo:block>
    </fo:block-container>
  </xsl:template>

  <!-- 翻訳抄録 -->
  <xsl:template match="trans-abstract">
    <fo:block-container xsl:use-attribute-sets="trans-abstract">
      <fo:block xsl:use-attribute-sets="reset-indents">
        <xsl:apply-templates/>
        <!-- 翻訳版キーワードリスト -->
        <fo:block xsl:use-attribute-sets="trans-keywords">
          <xsl:choose>
            <xsl:when test="$trans-lang='ja'">
              <xsl:text>キーワード：</xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <xsl:text>Key Words:&#x2002;</xsl:text>
            </xsl:otherwise>
          </xsl:choose>
          <!-- <xsl:text>Key Words:&#x2002;</xsl:text> -->
          <xsl:for-each select="../kwd-group[@xml:lang=$trans-lang]/kwd">
            <fo:inline>
              <xsl:apply-templates/>
            </fo:inline>
            <xsl:if test="not(position()=last())">
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:for-each>
          <xsl:text>.</xsl:text>
        </fo:block>
        <xsl:call-template name="banner-rule"/>
      </fo:block>
    </fo:block-container>
  </xsl:template>

  <!-- 連絡先情報など [jats-xslfo] -->
  <xsl:template name="set-correspondence-note">
    <xsl:for-each select="/article/front/article-meta">
      <fo:block span="none" space-after="0pt" space-after.precedence="force"
        font-size="0pt" line-height="0pt">
        <xsl:call-template name="make-footnote">
          <xsl:with-param name="contents">
            <fo:block padding-bottom="3pt">
              <xsl:apply-templates select="author-notes/fn"/>
              <xsl:for-each select="history">
                <fo:block>
                  <xsl:for-each select="date[@date-type='received']">
                    <xsl:value-of select="year"/>
                    <xsl:text>年</xsl:text>
                    <xsl:value-of select="month"/>
                    <xsl:text>月</xsl:text>
                    <xsl:value-of select="day"/>
                    <xsl:text>日受領　</xsl:text>
                  </xsl:for-each>
                  <xsl:for-each select="date[@date-type='accepted']">
                    <xsl:value-of select="year"/>
                    <xsl:text>年</xsl:text>
                    <xsl:value-of select="month"/>
                    <xsl:text>月</xsl:text>
                    <xsl:value-of select="day"/>
                    <xsl:text>日受理</xsl:text>
                  </xsl:for-each>
                </fo:block>
              </xsl:for-each>
              <xsl:apply-templates select="author-notes/corresp"/>
            </fo:block>
          </xsl:with-param>
        </xsl:call-template>
      </fo:block>
    </xsl:for-each>
  </xsl:template>


  <!-- セクション -->
  <xsl:template match="sec">
    <fo:block xsl:use-attribute-sets="section">
      <xsl:apply-templates select="title"/>
      <xsl:apply-templates select="sec-meta"/>
      <xsl:apply-templates mode="drop-title"/>
    </fo:block>
  </xsl:template>

  <!-- 大見出し -->
  <xsl:template match="body/sec/title">
    <fo:block xsl:use-attribute-sets="main-title">
      <xsl:call-template name="set-label-text">
        <xsl:with-param name="label-node" select="preceding-sibling::label"/>
      </xsl:call-template>
      <xsl:apply-templates/>
    </fo:block>
  </xsl:template>

  <!-- 中見出し -->
  <xsl:template match="body/*/sec/title" priority="3">
    <fo:block xsl:use-attribute-sets="section-title">
      <xsl:call-template name="set-label-text">
        <xsl:with-param name="label-node" select="preceding-sibling::label"/>
      </xsl:call-template>
      <xsl:apply-templates/>
    </fo:block>
  </xsl:template>

  <!-- 小見出し -->
  <xsl:template match="body/*/*/sec/title">
    <fo:block xsl:use-attribute-sets="subsection-title">
      <xsl:call-template name="set-label-text">
        <xsl:with-param name="label-node" select="preceding-sibling::label"/>
      </xsl:call-template>
      <xsl:apply-templates/>
    </fo:block>
  </xsl:template>

  <!-- 見出しのラベルを出力する -->
  <xsl:template name="set-label-text">
    <xsl:param name="label-node" select="label"/>
    <xsl:variable name="label">
      <xsl:apply-templates select="$label-node" mode="label-text"/>
    </xsl:variable>
    <xsl:if test="normalize-space($label)">
      <xsl:copy-of select="$label"/>
      <!-- ラベルの最後の文字が特定の全角文字以外のときスペースを入れる -->
      <xsl:if
        test="not(contains('　、。，．・：；）〕］｝〉》」』】',
        substring($label, string-length($label))))">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>
  </xsl:template>

  <!-- 見出しのラベル -->
  <xsl:template match="sec/label" mode="label-text">
    <fo:inline xsl:use-attribute-sets="section-label">
      <xsl:apply-templates/>
    </fo:inline>
  </xsl:template>

<xsl:template mode="contrib" match="aff">
	<fo:block xsl:use-attribute-sets="aff">
		<xsl:choose>
			<xsl:when test="child::label">
				<xsl:call-template name="set-label-text"/>
			</xsl:when>
			<xsl:otherwise>
				<!-- ラベルを持っていないときは、これの参照元の著者が持っているラベル（contrib/xref）の内容を利用する -->
				<xsl:variable name="ref-id" select="parent::aff-alternatives/@id" />
				<xsl:variable name="target-label">
					<xsl:value-of select="(parent::*/parent::contrib-group/contrib[@contrib-type='author']/xref[@ref-type = 'aff' and @rid = $ref-id])[1]" />
				</xsl:variable>
				<fo:inline xsl:use-attribute-sets="superscript">
					<xsl:value-of select="$target-label" />
					<xsl:choose>
						<xsl:when test="substring($target-label, string-length($target-label), 1) = ')'">
						</xsl:when>
						<xsl:otherwise>
							<!-- <xsl:text>)</xsl:text> -->
						</xsl:otherwise>
					</xsl:choose>
				</fo:inline>
			</xsl:otherwise>
		</xsl:choose>
		<xsl:apply-templates
			select="*|text()[normalize-space() or 
				not(preceding-sibling::*[1][self::label]) and 
				not(following-sibling::label)]"
		/>
	</fo:block>
</xsl:template>

  <xsl:template
    match="addr-line | country | fax | 
                       institution | phone">
    <!-- [jats-xslfo]で空白挿入していたのをやめる -->
    <!--<xsl:if test="preceding-sibling::*">
    <xsl:text> </xsl:text>
  </xsl:if>-->
    <xsl:apply-templates/>
  </xsl:template>


  <xsl:template match="fig/label | table-wrap/label" mode="label-text">
    <fo:float float="start">
      <fo:block xsl:use-attribute-sets="reset-indents caption-label">
        <xsl:apply-templates/>
      </fo:block>
    </fo:float>
  </xsl:template>

  <!--<xsl:template match="label" mode="label-text">
    <fo:float float="start">
      <fo:block xsl:use-attribute-sets="reset-indents">
        <xsl:apply-templates/>
      </fo:block>
    </fo:float>
  </xsl:template>-->


  <xsl:template match="fn-group/fn | table-wrap-foot//fn">
    <fo:block xsl:use-attribute-sets="endnote">
      <xsl:if test="label">
        <fo:float float="start">
          <fo:block xsl:use-attribute-sets="reset-indents">
            <xsl:apply-templates select="label" mode="label-text"/>
          </fo:block>
        </fo:float>
      </xsl:if>
      <xsl:apply-templates/>
    </fo:block>
  </xsl:template>

  <xsl:template match="author-notes/fn | author-notes/corresp">
    <fo:block xsl:use-attribute-sets="reset-indents">
      <xsl:apply-templates/>
    </fo:block>
  </xsl:template>

  <!-- 脚注のラベル -->
  <xsl:template match="fn/label" mode="label-text">
    <fo:inline xsl:use-attribute-sets="superscript">
      <xsl:apply-templates/>
    </fo:inline>
  </xsl:template>

  <!-- 所属のラベル -->
  <xsl:template match="aff/label" mode="label-text">
    <fo:inline xsl:use-attribute-sets="superscript">
      <xsl:apply-templates/>
    </fo:inline>
  </xsl:template>

  <!-- 英文所属名 -->
  <xsl:template match="aff[@xml:lang='en']/institution">
    <fo:inline font-style="italic">
      <xsl:apply-templates/>
    </fo:inline>
  </xsl:template>

  <!-- 英文抄録の段落 -->
  <xsl:template match="trans-abstract[@xml:lang='en']/p[1]">
    <fo:block xsl:use-attribute-sets="paragraph" text-indent="0em">
      <xsl:apply-templates/>
    </fo:block>
  </xsl:template>

  <!-- そのほか字下げしない段落 -->
  <xsl:template match="fn/p | caption/p | table-wrap-foot/p">
    <fo:block xsl:use-attribute-sets="paragraph" text-indent="0em">
      <xsl:apply-templates/>
    </fo:block>
  </xsl:template>

  <!-- 図のキャプション -->
  <xsl:template match="fig/caption">
    <fo:block xsl:use-attribute-sets="fig-caption">
      <xsl:apply-templates/>
    </fo:block>
  </xsl:template>

  <!-- 図 -->
  <xsl:template match="fig | fig-group">
    <xsl:param name="allow-float" select="true()"/>
    <xsl:call-template name="set-float">
      <xsl:with-param name="allow-float" select="$allow-float"/>
      <xsl:with-param name="contents">
        <fo:block-container xsl:use-attribute-sets="fig-box">
          <xsl:apply-templates select="@orientation"/>
          <fo:wrapper start-indent="0pc" end-indent="0pc">
            <xsl:apply-templates select="*[not(self::caption)]"/>
            <!-- ラベルとキャプションを下に移動 -->
            <xsl:apply-templates select="." mode="label"/>
            <xsl:apply-templates select="caption"/>
          </fo:wrapper>
        </fo:block-container>
      </xsl:with-param>
    </xsl:call-template>
  </xsl:template>

  <!-- 引用文献 -->
  <xsl:template match="ref">
    <fo:block xsl:use-attribute-sets="ref" keep-together.within-page="always">
      <xsl:apply-templates/>
    </fo:block>
  </xsl:template>

</xsl:stylesheet>
