﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：指定したページの指定した位置にPDFの1ページを挿入します。
  　　　ここまでに紹介したものと共通する処理が多いため、
  　　　今回に特有の処理についてdrawForm()メソッドにまとめています。

  Copyright (C) 2019- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/


using System;
using PdfTkNet;

namespace cookbook
{
    public class AddPdfOnPage
    {
        public static void Main(String[] args)
        {
            if (args.Length < 5)
            {
                Console.WriteLine("usage: AddPdfOnPage in-pdf-file out-pdf-file page-num insert-pdf-file insert-page-num align");
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    //コマンドライン引数の取得
                    int pageToAdd = int.Parse(args[2]);
                    String insertPdfURI = args[3];
                    int insertPageNum = int.Parse(args[4]);
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);
                    if ((numPages < 0) || (numPages < pageToAdd))
                    {
                        Console.WriteLine("page-numは入力PDFの全ページ数よりも小さい正の値を指定してください。");
                        return;
                    }
                    String align = args[5];

                    using (PtlPages pages = doc.getPages()) //ページコンテナの取得
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("入力PDFのページコンテナが空");
                            return;
                        }
                        // フォームの描画処理（）
                        using (PtlPage page = pages.get(pageToAdd - 1))// ページの取得(パラメータindexは0が先頭のため1を引く)
                        {
                            drawForm(page, insertPdfURI, insertPageNum, align);
                        }
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void drawForm(PtlPage page, String insertPdfURI, int insertPageNum, String align)
        {
            using (PtlContent content = page.getContent())// 挿入先ページコンテントの取得
            using (PtlParamInput insertPdf = new PtlParamInput(insertPdfURI)) // 挿入PDF指定に使うパラメータクラス
            using (PtlPDFDocument doc2 = new PtlPDFDocument()) // 挿入PDFの実体
            using (PtlRect outputRect = new PtlRect(setOutputRect())) // 出力矩形の設定（setOutputRectを用いて初期化）
            {
                doc2.load(insertPdf);
                int numPages = doc2.getPageCount();
                if ((numPages < 0) || (numPages < insertPageNum))
                {
                    Console.WriteLine("insert-page-numは挿入するPDFの全ページ数よりも小さい正の値を指定してください。");
                    Console.WriteLine("挿入PDFページ数：" + numPages);
                    return;
                }
                using (PtlPages pages2 = doc2.getPages())
                {
                    // ページコンテナが空かどうか
                    if (pages2.isEmpty())
                    {
                        Console.WriteLine("挿入するPDFのページコンテナが空");
                        return;
                    }
                    try
                    {
                        using (PtlPage pageInsert = pages2.get(insertPageNum - 1)) // ページの取得(パラメータindexは0が先頭のため1を引く))
                        {
                            // 指定ページの挿入
                            content.drawForm(outputRect, (PtlContent.ALIGN)Enum.Parse(typeof(PtlContent.ALIGN), align), pageInsert);
                        }
                    }
                    catch (ArgumentException ex)
                    {//PtlContent.ALIGNの指定が誤っていた場合のエラーメッセージ
                        Console.WriteLine(ex.Message);
                        Console.WriteLine("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                        Console.WriteLine(ex.StackTrace);
                    }
                }
            }
        }

        public static PtlRect setOutputRect()
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            PtlRect outputRect = new PtlRect();
            while (!isValueOkay)
            {
                Console.WriteLine("配置矩形の各数値を入力してください。");
                Console.Write("top (mm) : ");
                top = float.Parse(Console.ReadLine());
                Console.Write("bottom (mm) : ");
                bottom = float.Parse(Console.ReadLine());
                if (top < bottom) //不正矩形は再入力させる
                {
                    Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                    continue;
                }
                Console.Write("left (mm) : ");
                left = float.Parse(Console.ReadLine());
                Console.Write("right (mm) : ");
                right = float.Parse(Console.ReadLine());
                if (right < left) //不正矩形は再入力させる
                {
                    Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                    continue;
                }

                //矩形を正しく指定できた場合の処理
                isValueOkay = true;
                outputRect.setLeft(left);
                outputRect.setBottom(bottom);
                outputRect.setRight(right);
                outputRect.setTop(top);
            }
            return outputRect;
        }
    }
}
