﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：QRコード（PDF）を矩形で指定した位置のすべてのページに貼り付けます。

  Copyright (C) 2019- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/


using System;
using PdfTkNet;

namespace cookbook
{
    public class AddPdfQRcodeOnAllPage
    {
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                Console.WriteLine("usage: AddPdfQRcodeOnAllPage in-pdf-file out-pdf-file insert-QRcode-pdf");
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {

                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    //コマンドライン引数の取得
                    String insertPdfURI = args[2];
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);

                    using (PtlPages pages = doc.getPages())
                    using (PtlRect outputRect = setQROutputRect()) // 出力矩形の設定（setQROutputRect()を用いる）
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("入力PDFのページコンテナが空");
                            return;
                        }
                        for (int i = 0; i < numPages; i++)
                        {
                            using (PtlPage page = pages.get(i))// ページの取得
                            {
                                // フォームの描画処理
                                drawQRcode(page, insertPdfURI, outputRect);
                            }
                        }
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void drawQRcode(PtlPage page, String insertPdfURI, PtlRect outputRect)
        {
            using (PtlContent content = page.getContent())// 挿入先ページコンテントの取得
            using (PtlParamInput insertPdf = new PtlParamInput(insertPdfURI)) // 挿入PDF指定に使うパラメータクラス
            using (PtlPDFDocument doc2 = new PtlPDFDocument()) // 挿入PDFの実体
            {
                doc2.load(insertPdf);
                using (PtlPages pages2 = doc2.getPages())
                {
                    // ページコンテナが空かどうか
                    if (pages2.isEmpty())
                    {
                        Console.WriteLine("挿入するPDFのページコンテナが空");
                        return;
                    }
                    using (PtlPage pageInsert = pages2.get(0)) // 先頭ページの取得
                    {
                        // 指定ページの挿入（）
                        content.drawForm(outputRect, PtlContent.ALIGN.ALIGN_TOP_LEFT, pageInsert);
                    }
                }
            }
        }

        public static PtlRect setQROutputRect()
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            PtlRect outputRect = new PtlRect();
            while (!isValueOkay)
            {
                Console.WriteLine("QRコードの配置矩形の各数値を入力してください。");
                Console.WriteLine("矩形の左上座標が配置の基準になり、");
                Console.WriteLine("そこから右下座標までの範囲を超えない範囲でQRコードが描画されます。");
                Console.Write("top (mm) : ");
                top = float.Parse(Console.ReadLine());
                Console.Write("bottom (mm) : ");
                bottom = float.Parse(Console.ReadLine());
                if (top < bottom) //不正矩形は再入力させる
                {
                    Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                    continue;
                }
                Console.Write("left (mm) : ");
                left = float.Parse(Console.ReadLine());
                Console.Write("right (mm) : ");
                right = float.Parse(Console.ReadLine());
                if (right < left) //不正矩形は再入力させる
                {
                    Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                    continue;
                }

                //矩形を正しく指定できた場合の処理
                isValueOkay = true;
                outputRect.setLeft(left);
                outputRect.setBottom(bottom);
                outputRect.setRight(right);
                outputRect.setTop(top);
            }
            return outputRect;
        }
    }
}
