﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：指定したページの指定した位置に矩形を描きます。

  Copyright (C) 2019- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/


using System;
using PdfTkNet;


namespace cookbook
{
    public class AddRect
    {
        public static void Main(String[] args)
        {
            if (args.Length < 12)
            {
                Console.WriteLine("usage: AddRect in-pdf-file out-pdf-file page-num line-style line-width lineR lineG lineB fillR fillG fillB opacity");
                Console.WriteLine("line-style:\n 0: 実線, 1: 破線");
                Console.WriteLine("line-width:\n 0: 細い, 1: 中, 2:太い");
                return;
            }

            // line-styleの判定
            int lineStyle = int.Parse(args[3]);
            switch (lineStyle)
            {
                case 0:
                case 1:
                    break;
                default:
                    Console.WriteLine("line-styleには0か1の数値を指定してください。");
                    Console.WriteLine("usage: AddRect in-pdf-file out-pdf-file page-num line-style line-width colorR colorG colorB opacity");
                    Console.WriteLine("line-style:\n 0: 実線, 1: 破線");
                    Console.WriteLine("line-width:\n 0: 細い, 1: 中, 2:太い");
                    return;
            }

            // line-widthの判定
            int lineWidth = int.Parse(args[4]);
            switch (lineWidth)
            {
                case 0:
                case 1:
                case 2:
                    break;
                default:
                    Console.WriteLine("line-widthには0から2の数値を指定してください。");
                    Console.WriteLine("usage: AddRect in-pdf-file out-pdf-file page-num line-style line-width colorR colorG colorB opacity");
                    Console.WriteLine("line-style:\n 0: 実線, 1: 破線");
                    Console.WriteLine("line-width:\n 0: 細い, 1: 中, 2:太い");
                    return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    //コマンドライン引数の取得
                    int pageToAdd = int.Parse(args[2]);
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);
                    if ((numPages < 0) || (numPages < pageToAdd))
                    {
                        Console.WriteLine("page-numは入力PDFの全ページ数よりも小さい正の値を指定してください。");
                        return;
                    }
                    float lineR = float.Parse(args[5]);
                    float lineG = float.Parse(args[6]);
                    float lineB = float.Parse(args[7]);
                    float fillR = float.Parse(args[8]);
                    float fillG = float.Parse(args[9]);
                    float fillB = float.Parse(args[10]);
                    float opacity = float.Parse(args[11]);


                    using (PtlPages pages = doc.getPages()) //ページコンテナの取得
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("入力PDFのページコンテナが空");
                            return;
                        }
                        // フォームの描画処理（）
                        using (PtlPage page = pages.get(pageToAdd - 1))// ページの取得(パラメータindexは0が先頭のため1を引く)
                        {
                            drawRectOnPage(page, lineStyle, lineWidth, lineR, lineG, lineB, fillR, fillG, fillB, opacity);
                        }
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void drawRectOnPage(PtlPage page, int lineStyle, int lineWidth,
                                          float lineR, float lineG, float lineB,
                                          float fillR, float fillG, float fillB,
                                          float opacity)
        {
            using (PtlContent content = page.getContent()) // ページコンテントの取得
            using (PtlColorDeviceRGB lineRGB = new PtlColorDeviceRGB(lineR, lineG, lineB)) //色を指定
            using (PtlColorDeviceRGB fillRGB = new PtlColorDeviceRGB(fillR, fillG, fillB)) //色を指定
            using (PtlParamDrawShape paramDrawShape = new PtlParamDrawShape()) // 線の描画用パラメータクラス
            {
                Console.WriteLine("描画する矩形のページ位置を指定します。");
                using (PtlRect shapeRect = new PtlRect(setRect()))
                {
                    paramDrawShape.setLineColor(lineRGB); // 線の色を指定色に設定
                    paramDrawShape.setFillColor(fillRGB); // 塗りつぶしの色を指定色に設定
                    paramDrawShape.setOpacity(opacity); // 透明度を設定

                    // lineStyleの設定
                    switch (lineStyle)
                    {
                        case 0:
                            paramDrawShape.setLineStyle(PtlParamDrawShape.LINE_STYLE.LINE_STYLE_SOLID);
                            break;
                        case 1:
                            paramDrawShape.setLineStyle(PtlParamDrawShape.LINE_STYLE.LINE_STYLE_DASHED);
                            break;
                    }

                    // lineWidthの設定
                    switch (lineWidth)
                    {
                        case 0:
                            paramDrawShape.setLineWidth(PtlParamDrawShape.LINE_WIDTH.LINE_WIDTH_THIN);
                            break;
                        case 1:
                            paramDrawShape.setLineWidth(PtlParamDrawShape.LINE_WIDTH.LINE_WIDTH_MIDDLE);
                            break;
                        case 2:
                            paramDrawShape.setLineWidth(PtlParamDrawShape.LINE_WIDTH.LINE_WIDTH_THICK);
                            break;
                    }
                    // 指定ページの挿入
                    content.drawRect(shapeRect, paramDrawShape);
                }
            }
        }

        public static PtlRect setRect()
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            PtlRect rect = new PtlRect();
            while (!isValueOkay)
            {
                Console.WriteLine("矩形の各数値を入力してください。");
                Console.Write("top (mm) : ");
                top = float.Parse(Console.ReadLine());
                Console.Write("bottom (mm) : ");
                bottom = float.Parse(Console.ReadLine());
                if (top < bottom) //不正矩形は再入力させる
                {
                    Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                    continue;
                }
                Console.Write("left (mm) : ");
                left = float.Parse(Console.ReadLine());
                Console.Write("right (mm) : ");
                right = float.Parse(Console.ReadLine());
                if (right < left) //不正矩形は再入力させる
                {
                    Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                    continue;
                }

                //矩形を正しく指定できた場合の処理
                isValueOkay = true;
                rect.setLeft(left);
                rect.setBottom(bottom);
                rect.setRight(right);
                rect.setTop(top);
            }
            return rect;
        }
    }
}
