﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：指定したページの指定した位置に縦書きテキストを挿入します。
  　　　配置矩形のみ、プログラム内で指定します。

  Copyright (C) 2019- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/


using System;
using PdfTkNet;

namespace cookbook
{
    public class AddTextVertical
    {
        public static void Main(String[] args)
        {
            if (args.Length < 5)
            {
                Console.WriteLine("usage: AddTextVertical in-pdf-file out-pdf-file page-num text-to-add text-align");
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);
                    int pageToAdd = int.Parse(args[2]);
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);
                    if ((numPages < 0) || (numPages < pageToAdd))
                    {
                        Console.WriteLine("テキスト挿入ページは全ページ数よりも小さい正の値を指定してください。");
                        return;
                    }

                    String textToAdd = args[3];
                    String textAlign = args[4];
                    using (PtlPages pages = doc.getPages()) //ページコンテナの取得
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空");
                            return;
                        }

                        try
                        {
                            using (PtlPage page = pages.get(pageToAdd - 1))// ページの取得(パラメータindexは0が先頭のため1を引く)
                            using (PtlContent content = page.getContent())// ページコンテントの取得
                            using (PtlRect outputRect = setOutputRect())// 出力矩形の設定（setOutputRectを用いて初期化）
                            using (PtlParamFont font = new PtlParamFont())// フォント指定用パラメータクラス
                            using (PtlParamWriteString plainParam = new PtlParamWriteString()) // 文字描画のパラメータクラス。今回は何も設定しない。
                            {
                                plainParam.setFont(font); //デフォルトのフォントを設定。
                                                          // 文字列出力
                                content.writeStringV(outputRect, (PtlContent.ALIGN)Enum.Parse(typeof(PtlContent.ALIGN), textAlign), textToAdd, plainParam);
                            }
                        }
                        catch (ArgumentException ex) //PtlContent.ALIGNの指定が誤っていた場合のエラーメッセージ
                        {
                            Console.WriteLine(ex.Message);
                            Console.WriteLine("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                            Console.WriteLine(ex.StackTrace);
                        }
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static PtlRect setOutputRect()
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            PtlRect outputRect = new PtlRect();
            while (!isValueOkay)
            {
                Console.WriteLine("文字列の配置矩形の各数値を入力してください。");
                Console.Write("top (mm) : ");
                top = float.Parse(Console.ReadLine());
                Console.Write("bottom (mm) : ");
                bottom = float.Parse(Console.ReadLine());
                if (top < bottom)
                {
                    Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                    continue;
                }
                Console.Write("left (mm) : ");
                left = float.Parse(Console.ReadLine());
                Console.Write("right (mm) : ");
                right = float.Parse(Console.ReadLine());
                if (right < left)
                {
                    Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                    continue;
                }

                //矩形を正しく指定できた場合の処理
                isValueOkay = true;
                outputRect.setLeft(left);
                outputRect.setBottom(bottom);
                outputRect.setRight(right);
                outputRect.setTop(top);
            }
            return outputRect;
        }
    }
}
