﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：ページ分割

  Copyright (C) 2019- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/


using System;
using PdfTkNet;

namespace cookbook
{
    public class DividePage
    {
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                Console.WriteLine("usage: DividePage in-pdf-file out-pdf-file 分割方向");
                Console.WriteLine("分割方向\n0 : ページを上下に分割  1 : ページを左右に分割");
                return;
            }

            String direction = args[2];
            switch (direction)
            {
                case "0":
                case "1":
                    break;
                default:
                    Console.WriteLine("分割方向を正しく指定してください。");
                    Console.WriteLine("usage: DividePage in-pdf-file out-pdf-file 分割方向");
                    Console.WriteLine("分割方向\n0 : ページを上下に分割  1 : ページを左右に分割");
                    return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                using (PtlPDFDocument docNew = new PtlPDFDocument())
                using (PtlPages pagesNew = docNew.getPages())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // ページ数の取得
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);

                    using (PtlPages pages = doc.getPages()) //ページコンテナの取得
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空");
                            return;
                        }

                        for (int i = 0; i < numPages; i++)
                        {
                            // 読み込んだページの取得
                            using (PtlPage page = pages.get(i))
                            using (PtlSize size = page.getSize())
                            using (PtlRect viewBox = page.getViewBox()) // 読み込んだページの表示矩形取得
                            {
                                float width = size.getWidth();
                                float height = size.getHeight();

                                // ページを分割していくので注釈のLinkアクションを削除しておく
                                removeLinkAction(page);

                                // 読み込んだページを新規PDFに2ページ分追加
                                pagesNew.append(page, PtlPages.INSERT_OPTION.OPTION_NONE);
                                pagesNew.append(page, PtlPages.INSERT_OPTION.OPTION_NONE);

                                int numPagesNew = pagesNew.getCount();
                                using (PtlPage pageNew1 = pagesNew.get(numPagesNew - 2)) // 追加されたページを取得
                                using (PtlPage pageNew2 = pagesNew.get(numPagesNew - 1)) // 追加されたページを取得
                                using (PtlRect rect1 = new PtlRect(viewBox)) // 追加されたページに設定する表示矩形
                                using (PtlRect rect2 = new PtlRect(viewBox)) // 追加されたページに設定する表示矩形
                                {
                                    switch (direction)
                                    {
                                        case "0":
                                            {
                                                // 半分の高さ
                                                float halfHeight = size.getHeight() / 2;
                                                // 矩形の上半分
                                                rect1.setBottom(viewBox.getBottom() + halfHeight);
                                                // 矩形の下半分
                                                rect2.setTop(viewBox.getTop() - halfHeight);
                                            }
                                            break;
                                        case "1":
                                            {
                                                // 半分の幅
                                                float halfWidth = size.getWidth() / 2;
                                                // 矩形の左半分
                                                rect1.setRight(viewBox.getRight() - halfWidth);
                                                // 矩形の右半分
                                                rect2.setLeft(viewBox.getLeft() + halfWidth);
                                            }
                                            break;
                                    }
                                    // 追加されたページの表示矩形を設定
                                    pageNew1.setViewBox(rect1);
                                    pageNew2.setViewBox(rect2);
                                }
                            }
                        }
                    }

                    // ファイルに保存します。
                    docNew.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void removeLinkAction(PtlPage page)
        {
            if (!page.hasAnnots())
            {
                return;
            }
            using (PtlAnnots annots = page.getAnnots())
            {
                int numAnnots = annots.getCount();
                for (int i = 0; i < numAnnots; ++i)
                {
                    using (PtlAnnot annot = annots.get(i))
                    {
                        switch (annot.getType())
                        {
                            case PtlAnnot.ANNOT_TYPE.TYPE_LINK:
                                {
                                    using (PtlAnnotLink annotLink = (PtlAnnotLink)annot)
                                    using (PtlAction action = annotLink.getAction())
                                    {
                                        switch (action.getType())
                                        {
                                            case PtlAction.ACTION_TYPE.TYPE_GOTO:
                                                annotLink.removeAction();
                                                break;
                                            default:
                                                break;
                                        }
                                    }
                                }
                                break;
                            default:
                                break;
                        }
                    }
                }
            }
        }
    }
}
