/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：指定した画像について、
  　　　DPIを指定した場合と指定しなかった場合
  　　　それぞれの場合をページを変えて出力します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;


public class AddImageSetDpi {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            System.out.println("usage: java AddImageSetDpi in-pdf-file out-pdf-file insert-image-file dpi-to-set");
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            //コマンドライン引数の取得
            String imageURI = args[2];
            int numDPI = Integer.parseInt(args[3]);
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);
            if(numPages < 3)
            {
                System.out.println("ページ数が3未満です。"+ numPages +"ページ目まで出力されます。");
            }

            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("ページコンテナが空");
                    return;
                }
                for(int i = 0; (i < 3)&&(i < numPages); i++)
                {
                    try (PtlPage page = pages.get(i);// ページの取得(パラメータindexは0が先頭)
                         PtlContent content = page.getContent();// ページコンテントの取得
                         PtlRect outputRect = page.getViewBox(); // 出力矩形の設定。(ページの大きさを得るためにViewBoxを取得する)
                         PtlParamDrawImage paramDrawImage = new PtlParamDrawImage(); //画像の描画に使うパラメータクラス
                         PtlParamInput insertImage = new PtlParamInput(imageURI); //画像指定に使うパラメータクラス
                         PtlParamWriteString plainParam = new PtlParamWriteString()) // 文字描画のパラメータクラス。今回は何も設定しない。
                    {
                        //入力画像ストリームの設定
                        paramDrawImage.setImageStream(insertImage);
                        //マルチTiffのページ番号の設定（マルチTiffファイルにのみ有効） 
                        //数値を設定しない場合は、-1が設定されたとみなします
                        //PtlParamImagePageに設定する場合に限り、-1を設定すると全ページを挿入します。
                        paramDrawImage.setImagePageNumber(0);

                        // DPI入力設定。
                        switch(i){
                        case 0:
                            paramDrawImage.setUseOriginalDPI(false);
                            paramDrawImage.setDPI(numDPI);
                            // 画像出力
                            content.drawImage(outputRect, PtlContent.ALIGN.ALIGN_CENTER, paramDrawImage);
                            // DPIの状態テキストを描画
                            content.writeString(outputRect, PtlContent.ALIGN.ALIGN_CENTER, "setDpi : " + numDPI, plainParam);
                            break;

                        case 1:
                            paramDrawImage.setUseOriginalDPI(false);
                            // 画像出力
                            content.drawImage(outputRect, PtlContent.ALIGN.ALIGN_CENTER, paramDrawImage);
                            // DPIの状態テキストを描画
                            content.writeString(outputRect, PtlContent.ALIGN.ALIGN_CENTER, "Default DPI: 72" , plainParam);
                            break;

                        case 2:
                            paramDrawImage.setUseOriginalDPI(true);
                            // 画像出力
                            content.drawImage(outputRect, PtlContent.ALIGN.ALIGN_CENTER, paramDrawImage);
                            // DPIの状態テキストを描画
                            content.writeString(outputRect, PtlContent.ALIGN.ALIGN_CENTER, "Original DPI" , plainParam);
                            break;
                        }
                    }
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
}
