/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：指定したページの指定した始点・終点に線を引きます。
  　　　ここまでに紹介したものと共通する処理が多いため、
  　　　今回に特有の処理についてdrawLineOnPage()メソッドにまとめています。
  　　　始点・終点については実行時にプログラム内で指定します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;


public class AddLine {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 9) {
            System.out.println("usage: java AddLine in-pdf-file out-pdf-file page-num line-style line-width colorR colorG colorB opacity");
            System.out.println("line-style:\n 0: 実線, 1: 破線");
            System.out.println("line-width:\n 0: 細い, 1: 中, 2:太い");
            return;
        }

        // line-styleの判定
        int lineStyle = Integer.parseInt(args[3]);
        switch(lineStyle){
        case 0:
        case 1:
            break;
        default :
            System.err.println("line-styleには0か1の数値を指定してください。");
            System.err.println("usage: java AddLine in-pdf-file out-pdf-file page-num line-style line-width colorR colorG colorB opacity");
            System.err.println("line-style:\n 0: 実線, 1: 破線");
            System.err.println("line-width:\n 0: 細い, 1: 中, 2:太い");
            return;
        }

        // line-widthの判定
        int lineWidth = Integer.parseInt(args[4]);
        switch(lineWidth){
        case 0:
        case 1:
        case 2:
            break;
        default :
            System.err.println("line-widthには0から2の数値を指定してください。");
            System.err.println("usage: java AddLine in-pdf-file out-pdf-file page-num line-style line-width colorR colorG colorB opacity");
            System.err.println("line-style:\n 0: 実線, 1: 破線");
            System.err.println("line-width:\n 0: 細い, 1: 中, 2:太い");
            return;
        }
        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            //コマンドライン引数の取得
            int pageToAdd = Integer.parseInt(args[2]);
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);
            if((numPages < 0)||(numPages < pageToAdd)) {
                System.err.println("page-numは入力PDFの全ページ数よりも小さい正の値を指定してください。");
                return;
            }
            float colorR = Float.parseFloat(args[5]);
            float colorG = Float.parseFloat(args[6]);
            float colorB = Float.parseFloat(args[7]);
            float opacity = Float.parseFloat(args[8]);
            
            
            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("入力PDFのページコンテナが空");
                    return;
                }
                // フォームの描画処理（）
                try (PtlPage page = pages.get(pageToAdd - 1))// ページの取得(パラメータindexは0が先頭のため1を引く)
                {
                    drawLineOnPage(page, lineStyle, lineWidth, colorR, colorG, colorB, opacity, br);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void drawLineOnPage(PtlPage page, int lineStyle, int lineWidth, float colorR, float colorG, float colorB, float opacity, BufferedReader br) throws PtlException, Exception, Error{
        
        try(PtlPoint startPoint = new PtlPoint(); // 始点を表すクラス
            PtlPoint endPoint = new PtlPoint();   // 終点を表すクラス
            PtlContent content = page.getContent(); // ページコンテントの取得
            PtlColorDeviceRGB colorRGB = new PtlColorDeviceRGB(colorR, colorG, colorB); //色を指定
            PtlParamDrawShape paramDrawShape = new PtlParamDrawShape()) // 線の描画用パラメータクラス
        {
            System.out.println("線の始点を設定します。");
            setPoint(br, startPoint);
            System.out.println("線の終点を設定します。");
            setPoint(br, endPoint);


            paramDrawShape.setLineColor(colorRGB); // 線の色を指定色に設定
            paramDrawShape.setOpacity(opacity); // 透明度を設定

            // lineStyleの設定
            switch(lineStyle){
            case 0:
                paramDrawShape.setLineStyle(PtlParamDrawShape.LINE_STYLE.LINE_STYLE_SOLID);
                break;
            case 1:
                paramDrawShape.setLineStyle(PtlParamDrawShape.LINE_STYLE.LINE_STYLE_DASHED);
                break;
            }

            // lineWidthの設定
            switch(lineWidth){
            case 0:
                paramDrawShape.setLineWidth(PtlParamDrawShape.LINE_WIDTH.LINE_WIDTH_THIN);
                break;
            case 1:
                paramDrawShape.setLineWidth(PtlParamDrawShape.LINE_WIDTH.LINE_WIDTH_MIDDLE);
                break;
            case 2:
                paramDrawShape.setLineWidth(PtlParamDrawShape.LINE_WIDTH.LINE_WIDTH_THICK);
                break;
            }
            // 指定ページの挿入
            content.drawLine(startPoint, endPoint, paramDrawShape);
        }
    }

    public static void setPoint(BufferedReader br, PtlPoint pointToSet) throws IOException, PtlException, Exception, Error{
        System.out.println("x座標、y座標を入力してください。原点は用紙の左下です。");
        System.out.print("x (mm) : ");
        pointToSet.setX(Float.parseFloat(br.readLine()));
        System.out.print("y (mm) : ");
        pointToSet.setY(Float.parseFloat(br.readLine()));

    }

}
