/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：指定したページの指定した位置にPDFの1ページを挿入します。
  　　　ここまでに紹介したものと共通する処理が多いため、
  　　　今回に特有の処理についてdrawForm()メソッドにまとめています。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;


public class AddPdfOnPage {
    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.print("usage: java AddForm in-pdf-file out-pdf-file");
        System.out.println("page-num insert-pdf-file insert-page-num align");
    }
    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 5) {
            printUsage(); // usageメッセージの表示
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            //コマンドライン引数の取得
            int pageToAdd = Integer.parseInt(args[2]);
            String insertPdfURI = args[3];
            int insertPageNum = Integer.parseInt(args[4]);
            String align = args[5];
            // pageToAddの判定
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);
            if((numPages < 0)||(numPages < pageToAdd)) {
                System.err.println("page-numは入力PDFの全ページ数よりも小さい正の値を指定してください。");
                printUsage(); // usageメッセージの表示
                return;
            }

            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("入力PDFのページコンテナが空");
                    return;
                }
                // フォームの描画処理
                // 書き込み先ページを取得(パラメータindexは0が先頭のため1を引く)
                try (PtlPage page = pages.get(pageToAdd - 1))
                {
                    drawForm(page, insertPdfURI, insertPageNum, align, br);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void drawForm(PtlPage page, String insertPdfURI,
                                int insertPageNum, String align, BufferedReader br)
        throws PtlException, Exception, Error {
        try(PtlContent content = page.getContent();// 挿入先ページコンテントの取得
            PtlParamInput insertPdf = new PtlParamInput(insertPdfURI); // 挿入PDF指定に使うパラメータクラス
            PtlPDFDocument doc2 = new PtlPDFDocument(); // 挿入PDFの実体
            PtlRect outputRect = new PtlRect()) // 出力矩形の初期化
        {
            setRectCoordinate(br, outputRect);
            doc2.load(insertPdf);
            int numPages = doc2.getPageCount();
            if((numPages < 0)||(numPages < insertPageNum)) {
                System.err.println("insert-page-numは挿入するPDFの全ページ数よりも小さい正の値を指定してください。");
                System.out.println("挿入PDFページ数：" + numPages);
                return;
            }
            try(PtlPages pages2 = doc2.getPages();
                PtlPage page2 = pages2.get(insertPageNum - 1);
                PtlParamDrawForm formInsert = new PtlParamDrawForm();
                )
            {
                // ページコンテナが空かどうか
                if (pages2.isEmpty())
                {
                    System.out.println("挿入するPDFのページコンテナが空");
                    return;
                }
                // ページの取得(パラメータindexは0が先頭のため1を引く)
                formInsert.setPage(page2);
                // 指定ページの挿入
                //PtlContent.ALIGNの指定が誤っていた場合のエラーメッセージを出すためにtry文を用いる
                try{
                    content.drawForm(outputRect, PtlContent.ALIGN.valueOf(align), formInsert);
                }
                catch (IllegalArgumentException ex){
                    System.out.println(ex.getMessage());
                    System.out.println("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                    ex.printStackTrace();
                }
            }
        }
    }
    
    public static void setRectCoordinate(BufferedReader br, PtlRect outputRect)
                          throws IOException, PtlException, Exception, Error{
        float top, bottom, left, right;
        boolean isValueOkay = false;
        while(!isValueOkay) {
            System.out.println("配置矩形の各数値を入力してください。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) //不正矩形は再入力させる
            {
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            if(right < left) //不正矩形は再入力させる
            {
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setLeft(left);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setTop(top);
        }
    }

}
