/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：QRコード（PDF）を矩形で指定した位置のすべてのページに貼り付けます。
  　　　ここまでに紹介したものと共通する処理が多いため、
  　　　今回に特有の処理についてdrawForm()メソッドにまとめています。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;


public class AddPdfQRcodeOnAllPage {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            System.out.println("usage: java AddPdfQRcodeOnAllPage in-pdf-file out-pdf-file insert-QRcode-pdf");
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            //コマンドライン引数の取得
            String insertPdfURI = args[2];
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);

            try (PtlPages pages = doc.getPages();
                 PtlRect outputRect = new PtlRect()) {
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("入力PDFのページコンテナが空");
                    return;
                }
                setQROutputRect(br, outputRect); // 出力矩形の設定（setQROutputRect()を用いる）
                for(int i = 0; i < numPages; i++) {
                    try (PtlPage page = pages.get(i)) { // ページの取得
                        // フォームの描画処理
                        drawQRcode(page, insertPdfURI, outputRect);
                    }
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void drawQRcode (PtlPage page, String insertPdfURI, PtlRect outputRect)
        throws PtlException, Exception, Error {
        try(PtlContent content = page.getContent();// 挿入先ページコンテントの取得
            PtlParamInput insertPdf = new PtlParamInput(insertPdfURI); // 挿入PDF指定に使うパラメータクラス
            PtlPDFDocument doc2 = new PtlPDFDocument()) { // 挿入PDFの実体
            doc2.load(insertPdf);
            try(PtlPages pages2 = doc2.getPages()) {
                // ページコンテナが空かどうか
                if (pages2.isEmpty()) {
                    System.out.println("挿入するPDFのページコンテナが空");
                    return;
                }
                try(PtlPage pageInsert = pages2.get(0)) { // 先頭ページの取得
                    // 指定ページの挿入（）
                    content.drawForm(outputRect, PtlContent.ALIGN.ALIGN_TOP_LEFT, pageInsert);
                }
            }
        }
    }
    
    public static PtlRect setQROutputRect(BufferedReader br, PtlRect outputRect)
        throws IOException, PtlException, Exception, Error{
        float top, bottom, left, right;
        boolean isValueOkay = false;
        while(!isValueOkay) {
            System.out.println("QRコードの配置矩形の各数値を入力してください。");
            System.out.println("矩形の左上座標が配置の基準になり、");
            System.out.println("そこから右下座標までの範囲を超えない範囲でQRコードが描画されます。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) { //不正矩形は再入力させる
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            if(right < left) { //不正矩形は再入力させる
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setTop(top);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setLeft(left);
        }
        return outputRect;
    }

}
