/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：すべてのページの指定した位置にQRコードを挿入します。
  　　　配置矩形のみ、プログラム内で指定します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;


public class AddQRcodeOnAllPage {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            System.out.println("usage: java AddQRcodeOnAllPage in-pdf-file out-pdf-file insert-image-file");
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            //コマンドライン引数の取得
            String imageURI = args[2];
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);

            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("ページコンテナが空");
                    return;
                }
                try(PtlRect outputRect = new PtlRect())// 出力矩形の初期化
                {
                    setQROutputRect(br, outputRect); //QRコード入力を始動
                    for(int i = 0; i < numPages; i++)
                    {
                        try (PtlPage page = pages.get(i);// ページの取得(パラメータindexは0が先頭のため1を引く)
                             PtlContent content = page.getContent();// ページコンテントの取得
                             PtlParamDrawImage paramDrawImage = new PtlParamDrawImage(); //画像の描画に使うパラメータクラス
                             PtlParamInput insertImage = new PtlParamInput(imageURI)) //画像指定に使うパラメータクラス
                        {
                            //入力画像ストリームの設定
                            paramDrawImage.setImageStream(insertImage);

                            // 画像出力
                            content.drawImage(outputRect, PtlContent.ALIGN.ALIGN_TOP_LEFT, paramDrawImage);
                        }
                    }
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static PtlRect setQROutputRect(BufferedReader br,PtlRect outputRect)
           throws IOException, PtlException, Exception, Error{
        float top, bottom, left, right;
        boolean isValueOkay = false;
        while(!isValueOkay) {
            System.out.println("QRコードの配置矩形の各数値を入力してください。");
            System.out.println("矩形の左上座標が配置の基準になり、");
            System.out.println("そこから右下座標までの範囲を超えない範囲でQRコードが描画されます。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) //不正矩形は再入力させる
            {
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            if(right < left) //不正矩形は再入力させる
            {
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setTop(top);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setLeft(left);
        }
        return outputRect;
    }

}
