/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：指定したページの指定した位置に丸角矩形の描画をします。
  　　　ここまでに紹介したものと共通する処理が多いため、
  　　　今回に特有の処理についてdrawRoundRectOnPage()メソッドにまとめています。
  　　　矩形の指定・丸角部分のサイズ指定については実行時にプログラム内で行います。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;


public class AddRoundRect {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 12) {
            System.out.println("usage: java AddRoundRect in-pdf-file out-pdf-file page-num line-style line-width lineR lineG lineB fillR fillG fillB opacity");
            System.out.println("line-style:\n 0: 実線, 1: 破線");
            System.out.println("line-width:\n 0: 細い, 1: 中, 2:太い");
            return;
        }

        // line-styleの判定
        int lineStyle = Integer.parseInt(args[3]);
        switch(lineStyle){
        case 0:
        case 1:
            break;
        default :
            System.err.println("line-styleには0か1の数値を指定してください。");
            System.err.println("usage: java AddRoundRect in-pdf-file out-pdf-file page-num line-style line-width colorR colorG colorB opacity");
            System.err.println("line-style:\n 0: 実線, 1: 破線");
            System.err.println("line-width:\n 0: 細い, 1: 中, 2:太い");
            return;
        }

        // line-widthの判定
        int lineWidth = Integer.parseInt(args[4]);
        switch(lineWidth){
        case 0:
        case 1:
        case 2:
            break;
        default :
            System.err.println("line-widthには0から2の数値を指定してください。");
            System.err.println("usage: java AddRoundRect in-pdf-file out-pdf-file page-num line-style line-width colorR colorG colorB opacity");
            System.err.println("line-style:\n 0: 実線, 1: 破線");
            System.err.println("line-width:\n 0: 細い, 1: 中, 2:太い");
            return;
        }
        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            //コマンドライン引数の取得
            int pageToAdd = Integer.parseInt(args[2]);
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);
            if((numPages < 0)||(numPages < pageToAdd)) {
                System.err.println("page-numは入力PDFの全ページ数よりも小さい正の値を指定してください。");
                return;
            }
            float lineR = Float.parseFloat(args[5]);
            float lineG = Float.parseFloat(args[6]);
            float lineB = Float.parseFloat(args[7]);
            float fillR = Float.parseFloat(args[8]);
            float fillG = Float.parseFloat(args[9]);
            float fillB = Float.parseFloat(args[10]);
            float opacity = Float.parseFloat(args[11]);
            
            
            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("入力PDFのページコンテナが空");
                    return;
                }
                // フォームの描画処理（）
                try (PtlPage page = pages.get(pageToAdd - 1)) { // ページの取得(パラメータindexは0が先頭のため1を引く)
                    drawRoundRectOnPage(page, lineStyle, lineWidth, lineR, lineG, lineB, fillR, fillG, fillB, opacity, br);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void drawRoundRectOnPage(PtlPage page, int lineStyle, int lineWidth,
                                           float lineR, float lineG, float lineB,
                                           float fillR, float fillG, float fillB, 
                                           float opacity, BufferedReader br) throws PtlException, Exception, Error{
        try(PtlContent content = page.getContent(); // ページコンテントの取得
            PtlColorDeviceRGB lineRGB = new PtlColorDeviceRGB(lineR, lineG, lineB); //色を指定
            PtlColorDeviceRGB fillRGB = new PtlColorDeviceRGB(fillR, fillG, fillB); //色を指定
            PtlParamDrawShape paramDrawShape = new PtlParamDrawShape()) { // 線の描画用パラメータクラス
            System.out.println("描画する矩形のページ位置を指定します。");
            try(PtlRect shapeRect = new PtlRect()) {
                setRect(br, shapeRect);
                System.out.println("\n丸角の幅を入力してください。"); 
                System.out.print("width (mm) : "); 
                float width = Float.parseFloat(br.readLine());
                System.out.println("丸角の高さを入力してください。"); 
                System.out.print("height (mm) : "); 
                float height = Float.parseFloat(br.readLine());
                paramDrawShape.setLineColor(lineRGB); // 線の色を指定色に設定
                paramDrawShape.setFillColor(fillRGB); // 塗りつぶしの色を指定色に設定
                paramDrawShape.setOpacity(opacity); // 透明度を設定
                    
                // lineStyleの設定
                switch(lineStyle) {
                case 0:
                    paramDrawShape.setLineStyle(PtlParamDrawShape.LINE_STYLE.LINE_STYLE_SOLID);
                    break;
                case 1:
                    paramDrawShape.setLineStyle(PtlParamDrawShape.LINE_STYLE.LINE_STYLE_DASHED);
                    break;
                }

                // lineWidthの設定
                switch(lineWidth) {
                case 0:
                    paramDrawShape.setLineWidth(PtlParamDrawShape.LINE_WIDTH.LINE_WIDTH_THIN);
                    break;
                case 1:
                    paramDrawShape.setLineWidth(PtlParamDrawShape.LINE_WIDTH.LINE_WIDTH_MIDDLE);
                    break;
                case 2:
                    paramDrawShape.setLineWidth(PtlParamDrawShape.LINE_WIDTH.LINE_WIDTH_THICK);
                    break;
                }
                // 指定ページの挿入
                content.drawRoundRect(shapeRect, width, height, paramDrawShape);
            }
        }
    }

    public static void setRect(BufferedReader br, PtlRect rect)
                        throws IOException, PtlException, Exception, Error{
        float top, bottom, left, right;
        boolean isValueOkay = false;

        while(!isValueOkay) {
            System.out.println("矩形の各数値を入力してください。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) //不正矩形は再入力させる
            {
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            if(right < left) //不正矩形は再入力させる
            {
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            rect.setLeft(left);
            rect.setBottom(bottom);
            rect.setRight(right);
            rect.setTop(top);
        }
    }

}
