/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：文字列の色を設定して１ページ目中央に描画するプログラム。


  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;

public class AddTextSetColor {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 2) {
            System.out.println("usage: java AddTextSetColor in-pdf-file out-pdf-file color-space");
            System.out.println("color-space\n  0:無色 1:DeviceGray 2:DeviceRGB 3:DeviceCMYK");
            return;
        }

        int colorSpace = Integer.parseInt(args[2]);
        switch(colorSpace)
        {
        case 0:
        case 1:
        case 2:
        case 3:
            break;
        default:
            System.err.println("color-spaceには0から3の数を指定してください。");
            System.err.println("usage: java AddTextSetColor in-pdf-file out-pdf-file color-space");
            System.err.println("color-space\n  0:無色 1:DeviceGray 2:DeviceRGB 3:DeviceCMYK");
            return ;
        }
        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("ページコンテナが空");
                    return;
                }
                
                try (PtlPage page = pages.get(0); // ページの取得(先頭ページを取得)
                     PtlContent content = page.getContent(); // ページコンテントの取得
                     PtlRect outputRect = page.getViewBox(); // 出力矩形の設定。(ページの大きさを得るためにViewBoxを取得する)
                     PtlParamWriteString writeStringParam = new PtlParamWriteString(); // 文字描画のパラメータクラス
                     // 色表現用のクラスを作成
                     PtlColorNone colorNone = new PtlColorNone(); // 無色を表現したクラス
                     PtlColorDeviceGray colorGray = new PtlColorDeviceGray(); // DeviceGray色空間で色表現したクラス
                     PtlColorDeviceRGB colorRGB = new PtlColorDeviceRGB(); //DeviceRGB色空間で色表現したクラス
                     PtlColorDeviceCMYK colorCMYK = new PtlColorDeviceCMYK()) //DeviceCMYK色空間で色表現したクラス
                {

                    switch(colorSpace)
                    {
                    case 0: //無色
                        System.out.println("無色が指定されました。無色で文字を描画します。");
                        writeStringParam.setTextColor(colorNone);
                        break;
                    case 1: //DeviceGray
                        System.out.println("グレースケール値を0.0～1.0の範囲で入力してください。0.0が黒、1.0が白。");
                        colorGray.setGray(setValue(br));
                        writeStringParam.setTextColor(colorGray);
                        break;
                    case 2: //DeviceRGB
                        System.out.println("RGBの各要素の値を0.0～1.0の範囲で入力してください。0.0が要素なし、1.0が要素の最大");
                        System.out.print("赤：");
                        colorRGB.setR(setValue(br));
                        System.out.print("緑：");
                        colorRGB.setG(setValue(br));
                        System.out.print("青：");
                        colorRGB.setB(setValue(br));
                        writeStringParam.setTextColor(colorRGB);
                        break;
                    case 3: //DeviceCMYK
                        System.out.println("CYMKの各要素の値を0.0～1.0の範囲で入力してください。0.0が要素なし、1.0が要素の最大");
                        System.out.print("シアン：");
                        colorCMYK.setC(setValue(br));
                        System.out.print("マゼンタ：");
                        colorCMYK.setM(setValue(br));
                        System.out.print("黄：");
                        colorCMYK.setY(setValue(br));
                        System.out.print("色調(黒)：");
                        colorCMYK.setK(setValue(br));
                        writeStringParam.setTextColor(colorCMYK);
                        break;                        
                    }

                    // 文字列出力 (中央に文字列を作成)
                    content.writeString(outputRect, PtlContent.ALIGN.ALIGN_CENTER,
                                        "writeString Test : set color", writeStringParam);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static float setValue(BufferedReader br)throws IOException{
        float value = -1.0f;
        boolean isValueOk = false;
        while(!isValueOk)
        {
            value = Float.parseFloat(br.readLine());
            if((value < 0) || (1.0f < value))
            {
                System.out.println("0.0から1.0の範囲で再度値を設定してください。");
            }
            else {
                isValueOk = true;
            }
        }
        return value;

    }
}
