/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：PDF文書の4ページを1ページに割り付け(4in1）します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class AllocatePages {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 2) {
            System.out.println("usage: java AllocatePages in-pdf-file out-pdf-file");
            return;
        }

        float width = 210.0f;  // A4横長
        float height = 297.0f; // A4縦長
        
        PtlRect rect[] = null;
        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             PtlPDFDocument docNew = new PtlPDFDocument())
        {
            // PDFファイルをロード
            doc.load(inputFile);

            try (PtlPages pagesNew = docNew.getPages(); //新規PDFページコンテナの取得
                 PtlPages pages = doc.getPages(); //入力PDFページコンテナの取得
                 PtlRect rectA4V = new PtlRect(0.0f, 0.0f, width, height))// A4縦用紙
            {
                rect = new PtlRect[] {new PtlRect(0.0f, height/2, width/2, height),  // 上左
                                      new PtlRect(width/2, height/2, width, height), // 上右
                                      new PtlRect(0.0f, 0.0f, width/2, height/2),    // 下左
                                      new PtlRect(width/2, 0.0f, width, height/2)};  // 下右
                int numPages = doc.getPageCount();
                int pageNumber = 0;
                int i = 0;

                while (i<numPages) {
                    try (PtlPage pageTemp = new PtlPage()) // 割り付けを行うA4縦ページ
                    {
                        // 割り付けを行うA4縦ページの追加
                        pagesNew.append(pageTemp, PtlPages.OPTION_NONE);

                        // 追加したページの取得
                        try (PtlPage pageNew = pagesNew.get(pageNumber))
                        {
                            pageNew.setViewBox(rectA4V);

                            // 割り付けを行うA4縦ページのコンテント取得
                            try (PtlContent content = pageNew.getContent())
                            {
                                // 4ページを1ページに割り付ける
                                for(int j=0; j<4 && i<numPages; ++j,++i) {
                                    // 割り付けするページ
                                    try (PtlPage page = pages.get(i))
                                    {
                                        // ページ割り付け
                                        content.drawForm(rect[j], PtlContent.ALIGN.ALIGN_CENTER, page);
                                    }
                                }
                            }
                        }
                        ++pageNumber;
                    }
                }
            }
            finally {
                if ( rect != null ) 
                {
                    for(int i=0; i<4; ++i) {
                        rect[i].close();
                    }
                }
            }
            // ファイルに保存します。
            docNew.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
}
