/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：ページ分割

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class DividePage {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            System.out.println("usage: java DividePage in-pdf-file out-pdf-file 分割方向");
            System.out.println("分割方向\n0 : ページを上下に分割  1 : ページを左右に分割");
            return;
        }

        String direction = args[2];
        switch (direction) {
        case "0":
        case "1":
            break;
        default:
            System.err.println("分割方向を正しく指定してください。");
            System.err.println("usage: java DividePage in-pdf-file out-pdf-file 分割方向");
            System.err.println("分割方向\n0 : ページを上下に分割  1 : ページを左右に分割");
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             PtlPDFDocument docNew = new PtlPDFDocument();
             PtlPages pagesNew = docNew.getPages())
        {
            // PDFファイルをロードします。
            doc.load(inputFile);

            // ページ数の取得
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);

            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("ページコンテナが空");
                    return;
                }

                for (int i = 0; i < numPages; i++)
                {
                    // 読み込んだページの取得
                    try (PtlPage page = pages.get(i);
                         PtlSize size = page.getSize();
                         PtlRect viewBox = page.getViewBox()) // 読み込んだページの表示矩形取得
                    {
                        // ページを分割していくので注釈のLinkアクションを削除しておく
                        removeLinkAction(page);

                        // 読み込んだページを新規PDFに2ページ分追加
                        pagesNew.append(page, PtlPages.OPTION_NONE);
                        pagesNew.append(page, PtlPages.OPTION_NONE);

                        int numPagesNew = pagesNew.getCount();
                        try (PtlPage pageNew1 = pagesNew.get(numPagesNew - 2); // 追加されたページを取得
                             PtlPage pageNew2 = pagesNew.get(numPagesNew - 1); // 追加されたページを取得
                             PtlRect rect1 = new PtlRect(viewBox); // 追加されたページに設定する表示矩形
                             PtlRect rect2 = new PtlRect(viewBox)) // 追加されたページに設定する表示矩形
                        {
                            switch (direction)
                            {
                            case "0":
                            {
                                // 半分の高さ
                                float halfHeight = size.getHeight() / 2;
                                // 矩形の上半分
                                rect1.setBottom(viewBox.getBottom() + halfHeight);
                                // 矩形の下半分
                                rect2.setTop(viewBox.getTop() - halfHeight);
                            }
                            break;
                            case "1":
                            {
                                // 半分の幅
                                float halfWidth = size.getWidth() / 2;
                                // 矩形の左半分
                                rect1.setRight(viewBox.getRight() - halfWidth);
                                // 矩形の右半分
                                rect2.setLeft(viewBox.getLeft() + halfWidth);
                            }
                            break;
                            }
                            // 追加されたページの表示矩形を設定
                            pageNew1.setViewBox(rect1);
                            pageNew2.setViewBox(rect2);
                        }
                    }
                }
            }

            // ファイルに保存します。
            docNew.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void removeLinkAction(PtlPage page)
                       throws PtlException, Exception, Error {
        if (!page.hasAnnots()) {
            return;
        }
        try (PtlAnnots annots = page.getAnnots())
        {
            int numAnnots = annots.getCount();
            for(int i=0; i<numAnnots; ++i) {
                try (PtlAnnot annot = annots.get(i))
                {
                    switch (annot.getType()) {
                    case TYPE_LINK:
                    {
                        try (PtlAnnotLink annotLink = (PtlAnnotLink)annot;
                             PtlAction action = annotLink.getAction())
                        {
                            switch (action.getType()) {
                            case TYPE_GOTO:
                                annotLink.removeAction();
                                break;
                            default:
                                break;
                            }
                        }
                        break;
                    }
                    
                    default:
                        break;
                    }
                }
            }
        }
    }
}
