/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：コンテントの重なりを例示するサンプルプログラムです。
  　　　指定したPDFの1ページ目に矩形・画像・PDF・文字列の順に描画し、
  　　　あとから描画したものが前にくることを示します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;


public class DrawOverlappedContents {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            System.out.println("usage: java DrawOverlappedContents in-pdf-file out-pdf-file insert-image-file insert-pdf-file");
            return;
        }
        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            //コマンドライン引数の取得
            String imageURI = args[2];
            String insertPdfURI = args[3];

            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("入力PDFのページコンテナが空");
                    return;
                }
                // フォームの描画処理（）
                try (PtlPage page = pages.get(0);
                     PtlSize size = page.getSize())// 先頭ページの取得
                {
                    float halfHeight = size.getHeight() /2;
                    float halfWidth = size.getWidth() / 2;
                    drawRectAtCenter(page, halfHeight, halfWidth);
                    drawTextAtCenter(page, halfHeight, halfWidth);
                    drawPdfAtCenter(page, insertPdfURI, halfHeight, halfWidth);
                    drawImageAtCenter(page, imageURI, halfHeight, halfWidth);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void drawRectAtCenter(PtlPage page, float halfHeight, float halfWidth) throws PtlException, Exception, Error{
        try(PtlContent content = page.getContent(); // ページコンテントの取得
            PtlRect shapeRect = page.getViewBox(); // 描画範囲矩形
            PtlColorDeviceRGB lineRGB = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f); //線の描画色を指定（赤色）
            PtlColorDeviceRGB fillRGB = new PtlColorDeviceRGB(1.0f, 1.0f, 0.0f); //塗りつぶし範囲の色を指定（黄色）
            PtlParamDrawShape paramDrawShape = new PtlParamDrawShape()) // 線の描画用パラメータクラス
        {
            // 描画位置の調整（中心付近で重なり合わせる。）
            shapeRect.setLeft(shapeRect.getLeft() + (halfWidth / 2));
            shapeRect.setBottom(shapeRect.getBottom() + halfHeight - 10.0f);
            shapeRect.setRight(shapeRect.getRight() - halfWidth + 10.0f);
            shapeRect.setTop(shapeRect.getTop() - (halfHeight / 2));

            paramDrawShape.setLineColor(lineRGB); // 線の色を指定色に設定
            paramDrawShape.setFillColor(fillRGB); // 塗りつぶしの色を指定色に設定
                    
            // 線の太さ・形式を決定
            paramDrawShape.setLineStyle(PtlParamDrawShape.LINE_STYLE.LINE_STYLE_SOLID);
            paramDrawShape.setLineWidth(PtlParamDrawShape.LINE_WIDTH.LINE_WIDTH_THICK);
            // 指定ページの挿入
            content.drawRect(shapeRect, paramDrawShape);
        }
    }


    public static void drawImageAtCenter(PtlPage page, String imageURI, float halfHeight, float halfWidth) throws IOException, PtlException, Exception, Error{
        try(PtlContent content = page.getContent();
            PtlRect outputRect = page.getViewBox();
            PtlParamInput insertImage = new PtlParamInput(imageURI);
            PtlParamDrawImage paramDrawImage = new PtlParamDrawImage())
        {
            //出力位置の調整（中心付近で重なり合わせる。）
            outputRect.setLeft(outputRect.getLeft() + halfWidth - 10.0f);
            outputRect.setBottom(outputRect.getBottom() + halfHeight - 10.0f);
            outputRect.setRight(outputRect.getRight() - (halfWidth / 2));
            outputRect.setTop(outputRect.getTop() - (halfHeight / 2));

            //入力画像ストリームの設定
            paramDrawImage.setImageStream(insertImage);
            //マルチTiffのページ番号の設定（マルチTiffファイルにのみ有効） 
            //数値を設定しない場合は、-1が設定されたとみなします
            //PtlParamImagePageに設定する場合に限り、-1を設定すると全ページを挿入します。
            paramDrawImage.setImagePageNumber(0);
            
            // 画像出力
            content.drawFitImage(outputRect, PtlContent.ALIGN.ALIGN_BOTTOM_LEFT, paramDrawImage);
        }
    }

    public static void drawPdfAtCenter(PtlPage page, String insertPdfURI, float halfHeight, float halfWidth) throws PtlException, Exception, Error{
        try(PtlContent content = page.getContent();// 挿入先ページコンテントの取得
            PtlRect outputRect = page.getViewBox(); // 出力矩形
            PtlParamInput insertPdf = new PtlParamInput(insertPdfURI); // 挿入PDF指定に使うパラメータクラス
            PtlPDFDocument doc2 = new PtlPDFDocument()) // 挿入PDFの実体
        {
            //出力位置の調整（中心付近で重なり合わせる。）
            outputRect.setLeft(outputRect.getLeft() + halfWidth);
            outputRect.setBottom(outputRect.getBottom() + (halfHeight / 2));
            outputRect.setRight(outputRect.getRight() - (halfWidth / 2));
            outputRect.setTop(outputRect.getTop() - halfHeight + 10.0f);

            doc2.load(insertPdf);
            try(PtlPages pages2 = doc2.getPages())
            {
                // ページコンテナが空かどうか
                if (pages2.isEmpty())
                {
                    System.out.println("挿入するPDFのページコンテナが空");
                    return;
                }
                try(PtlPage pageInsert = pages2.get(0)) // 先頭ページの取得
                {
                    // 指定ページの挿入
                    content.drawForm(outputRect, PtlContent.ALIGN.ALIGN_TOP_LEFT, pageInsert);
                }
            }
        }
    }

    public static void drawTextAtCenter(PtlPage page, float halfHeight, float halfWidth) throws IOException, PtlException, Exception, Error{
        try(PtlContent content = page.getContent();
            PtlRect outputRect = page.getViewBox();
            PtlParamWriteString writeStringParam = new PtlParamWriteString(); // 文字描画のパラメータクラス。今回は何も設定しない
            PtlColorDeviceRGB textColor = new PtlColorDeviceRGB(0.0f, 0.0f, 1.0f); // 文字の描画色を指定（青色）
            PtlParamFont font = new PtlParamFont())
        {

            //出力位置の調整（中心付近で重なり合わせる。）
            outputRect.setLeft(outputRect.getLeft() + (halfWidth / 2));
            outputRect.setBottom(outputRect.getBottom() + (halfHeight / 2));
            outputRect.setRight(outputRect.getRight() - halfWidth + 10.0f);
            outputRect.setTop(outputRect.getTop() - halfHeight + 10.0f);

            //要素指定
            font.setSize(24.0f);
            writeStringParam.setFont(font);
            writeStringParam.setTextColor(textColor);
                                    
            // テキストの出力
            content.writeString(outputRect, PtlContent.ALIGN.ALIGN_TOP_RIGHT, "これはテキストです", writeStringParam);
        }
    }
    
}
