﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：PDF 文書の指定したページにPDF透かしを挿入します。
  　　　指定したPDFの指定したページを読み取ってそれを透かしとします。
  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class ColorWaterMarkAppend
    {
        // そのクラスのusageを表示する関数
        public static void printUsage()
        {
            Console.Write("usage: ColorWaterMarkAppend in-pdf-file out-pdf-file");
            Console.WriteLine(" colorR colorG colorB");
            Console.WriteLine("color-R, color-G, color-B :");
            Console.WriteLine("0.0～1.0の数値で指定。それぞれの色について1.0が最大");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                printUsage(); // usageメッセージの表示
                return;
            }
            // コマンドライン引数の読み取り
            float colorR = float.Parse(args[2]);
            float colorG = float.Parse(args[3]);
            float colorB = float.Parse(args[4]);

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);
                    // 透かしの追加
                    appendColorWaterMark(doc, colorR, colorG, colorB);
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void appendColorWaterMark(PtlPDFDocument doc, float colorR,
                                                float colorG, float colorB)
        {
            using (PtlParamWaterMarkColor waterMarkColor = new PtlParamWaterMarkColor())
            {
                //透かしを配置する矩形の設定
                waterMarkColor.setMargin(10.0f, 10.0f, 10.0f, 10.0f);
                //透かしに指定する色の設定
                using (PtlColorDeviceRGB color = new PtlColorDeviceRGB(colorR, colorG, colorB))
                {
                    waterMarkColor.setColor(color);
                }

                // 以下、PDF透かし以外と共通するデフォルトの設定
                // 透かしの名前の設定
                waterMarkColor.setName("waterMarkColor");
                // 透かしの配置の設定
                waterMarkColor.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
                // 透かしのZオーダーの設定 ZORDER_FRONT = 1
                waterMarkColor.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
                // 透かしの不透明度の設定
                waterMarkColor.setOpacity(0.5f);
                // デフォルトの設定終わり

                // 透かしを文書に設定する
                doc.appendWaterMark(waterMarkColor);
            }
        }
    }
}
