﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：利用の制限
  　　　内容のコピー関する制限を設定します。
  　　　暗号化済みのファイルの場合はその他のセキュリティ情報は保持されます。
  　　　40 bit RC4 暗号化の場合はコピーの可否とアクセシビリティ有効の可否が連動します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class EncryptSetAccessibility
    {
        public static void printUsage()
        {
            Console.Write("usage: EncryptSetAccessibility in-pdf-file out-pdf-file ");
            Console.WriteLine("アクセシビリティ有効の可否 out-pdf-ownerpass [in-pdf-password]");
            Console.Write("アクセシビリティ（スクリーンリーダーデバイスのテキストアクセス）を");
            Console.WriteLine("有効にするか");
            Console.WriteLine("0 : 有効にしない 1 : 有効にする");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 4)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の取得
            bool permAccessibility = false;
            try
            {
                switch (int.Parse(args[2]))
                {
                    case 0:
                        permAccessibility = false;
                        Console.WriteLine("この文書のアクセシビリティ（スクリーンリーダーデバイスのテキストアクセス）を有効にしません。");
                        break;
                    case 1:
                        permAccessibility = true;
                        Console.WriteLine("この文書のアクセシビリティ（スクリーンリーダーデバイスのテキストアクセス）を有効にします。");
                        break;
                    default:
                        Console.WriteLine("アクセシビリティ有効の可否は、0か1の数値で指定してください。");
                        printUsage(); // usageメッセージの表示
                        return;
                }
            }
            catch (FormatException)
            {
                Console.WriteLine("アクセシビリティ有効の可否は、0か1の数値で指定してください。");
                printUsage(); // usageメッセージの表示
                return;
            }
            String outOwnerPass = args[3];

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    if (args.Length > 4)
                    {
                        String password = (String)args[4];
                        // パスワードのセット
                        doc.setPassword(password);
                    }

                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // 暗号化の有無判定
                    if (doc.isEncrypted())
                    {
                        // 暗号化情報の取得
                        setPermAccessibility(doc, outOwnerPass, permAccessibility);
                    }
                    else
                    {
                        Console.WriteLine("暗号化されていないファイルです。\n");
                        Console.WriteLine("アクセシビリティ有効の可否を設定して暗号化します。");
                        Console.WriteLine("暗号化種類の設定をします。1から3の数値を指定してください。");
                        Console.WriteLine("それ以外の数値が設定された場合は 3 が選択されたものとして扱います。");
                        Console.WriteLine("暗号化種類\n1 : 128 bit RC4  2 : 128 bit AES  3 : 256 bit AES");
                        int kind;
                        try
                        {
                            kind = int.Parse(Console.ReadLine());
                            // セキュリティ設定
                            switch (kind)
                            {
                                case 1:
                                    // 128 bit RC4
                                    encrypt128RC4setAccessibility(doc, outOwnerPass, permAccessibility);
                                    break;
                                case 2:
                                    // 128 bit AES
                                    encrypt128AESsetAccessibility(doc, outOwnerPass, permAccessibility);
                                    break;
                                case 3:
                                default:
                                    // 256 bit AES
                                    encrypt256AESsetAccessibility(doc, outOwnerPass, permAccessibility);
                                    break;
                            }
                        }
                        catch (FormatException nfe)
                        {
                            Console.WriteLine("暗号化の種類指定は整数を用いてください。");
                            Console.WriteLine(nfe.StackTrace);
                        }

                    }
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }


        public static void encrypt128RC4setAccessibility(PtlPDFDocument doc, String outOwnerPass,
                                                         bool permAccessibility)
        {
            // 128 bit RC4
            Console.WriteLine("128 bit RC4暗号化が選択されました。");
            using (PtlEncryptStandard128RC4 enc128r = new PtlEncryptStandard128RC4())
            {
                setAccessibilityWithDefaultPermission(doc, outOwnerPass, permAccessibility, enc128r);
            }
        }

        public static void encrypt128AESsetAccessibility(PtlPDFDocument doc, String outOwnerPass,
                                                         bool permAccessibility)
        {
            // 128 bit AES
            Console.WriteLine("128 bit AES暗号化が選択されました。");
            using (PtlEncryptStandard128AES enc128a = new PtlEncryptStandard128AES())
            {
                setAccessibilityWithDefaultPermission(doc, outOwnerPass, permAccessibility, enc128a);
            }
        }

        public static void encrypt256AESsetAccessibility(PtlPDFDocument doc, String outOwnerPass,
                                                         bool permAccessibility)
        {
            // 256 bit AES
            Console.WriteLine("256 bit AES暗号化が選択されました。");
            using (PtlEncryptStandard256AES enc256 = new PtlEncryptStandard256AES())
            {
                setAccessibilityWithDefaultPermission(doc, outOwnerPass, permAccessibility, enc256);
            }
        }

        public static void setAccessibilityWithDefaultPermission(PtlPDFDocument doc, String outOwnerPass,
                                                                 bool permAccessibility,
                                                                 PtlEncryptStandard enc)
        {
            // 暗号化するコンテンツ
            // 文書の全てのコンテンツを暗号化します。
            enc.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL);
            // オーナーパスワード値の設定
            enc.setOwnerPassword(outOwnerPass);
            
            using (PtlEncryptPermissionType2 perms2 = new PtlEncryptPermissionType2())
            {
                // 印刷権限の設定
                // PERM_PRINT_HIGH : 高解像度での印刷を許可
                perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_HIGH);

                // 変更権限の設定
                // ページ抽出を除く全ての動作を許可
                perms2.setModify(PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_MODYFYDOC);

                // 内容のコピーを有効にするかどうかの設定
                perms2.setCopy(false);

                // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの設定
                // true: 有効にする
                perms2.setAccessibility(permAccessibility);

                // ユーザアクセス許可フラグの設定
                enc.setPermission(perms2);
            }

            // 暗号化情報の設定
            doc.setEncrypt(enc);

        }

        public static void setPermAccessibility(PtlPDFDocument doc, String outOwnerPass,
                                                bool permAccessibility)
        {
            using (PtlEncrypt encrypt = doc.getEncrypt())
            {

                // 標準セキュリティハンドラのメソッド取得
                if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_STANDARD)
                {
                    // getFilterType()に判定済みなので PtlEncryptStandard にダウンキャストする。
                    PtlEncryptStandard encryptStandard = (PtlEncryptStandard)encrypt;
                    encryptStandard.setOwnerPassword(outOwnerPass);
                    // ユーザアクセス許可フラグの取得
                    using (PtlEncryptPermission perms = encryptStandard.getPermission())
                    {
                        // タイプ取得
                        switch (perms.getType())
                        {
                            case PtlEncryptPermission.PERM_TYPE.PERM_TYPE1:
                                // Type 1は40bit RC4なので、128bit RC4で新規に権限を設定する。
                                Console.WriteLine("入力ファイルは40 bit RC4暗号で暗号化されています。");
                                Console.WriteLine("新規に暗号化します。");
                                encrypt128RC4setAccessibility(doc, outOwnerPass, permAccessibility);
                                break;

                            case PtlEncryptPermission.PERM_TYPE.PERM_TYPE2:
                                // getType()で判定されたのでpermsをダウンキャストできる。
                                PtlEncryptPermissionType2 perms2 = (PtlEncryptPermissionType2)perms;
                                // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの設定
                                if(permAccessibility)
                                {
                                	//有効にする場合：コピー禁止設定の有無はそのまま引き継ぐ
                                	perms2.setAccessibility(permAccessibility);
                                	encryptStandard.setPermission(perms2);
                                	break;
                                }
                                else
                                {
                                	//無効にする場合：コピー禁止の設定を行う
                                	perms2.setCopy(false);
                                	perms2.setAccessibility(permAccessibility);
                                	encryptStandard.setPermission(perms2);
                                	break;
                                }
                        }
                    }
                    // 暗号化情報の設定
                    doc.setEncrypt(encryptStandard);
                }
                else
                {
                    Console.WriteLine("標準セキュリティハンドラ以外のセキュリティハンドラが使われたPDFです。");
                    Console.WriteLine("このPDFの権限を変更することはできません。");
                    return;
                }
            }
        }
    }
}
