﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：編集不可セキュリティの設定
  　　　編集の可否について設定します。
  　　　暗号化済みのファイルの場合はその他のセキュリティ情報は保持されます。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class EncryptSetModify
    {
        /**
     * @param args the command line arguments
     */
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                Console.Write("usage: EncryptSetModify in-pdf-file out-pdf-file ");
                Console.WriteLine("out-pdf-ownerpass [in-pdf-password]");
                return;
            }

            // コマンドライン引数の取得
            String outOwnerPass = args[2];

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    if (args.Length > 3)
                    {
                        String password = (String)args[3];
                        // パスワードのセット
                        doc.setPassword(password);
                    }

                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // 暗号化の有無判定
                    if (doc.isEncrypted())
                    {
                        // 暗号化情報の取得
                        setPermModify(doc, outOwnerPass);
                    }
                    else
                    {
                        Console.WriteLine("暗号化されていないファイルです。\n");
                        Console.WriteLine("変更権限を設定して暗号化します。");
                        Console.WriteLine("暗号化種類の設定をします。1から3の数値を指定してください。");
                        Console.WriteLine("それ以外の数値が設定された場合は 3 が選択されたものとして扱います。");
                        Console.WriteLine("暗号化種類");
                        Console.WriteLine("1 : 128 bit RC4  2 : 128 bit AES  3 : 256 bit AES");
                        int kind = int.Parse(Console.ReadLine());
                        // セキュリティ設定
                        switch (kind)
                        {
                            case 1:
                                // 128 bit RC4
                                encrypt128RC4setModify(doc, outOwnerPass);
                                break;
                            case 2:
                                // 128 bit AES
                                encrypt128AESsetModify(doc, outOwnerPass);
                                break;
                            case 3:
                            default:
                                // 256 bit AES
                                encrypt256AESsetModify(doc, outOwnerPass);
                                break;
                        }

                    }
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void encrypt128RC4setModify(PtlPDFDocument doc, String outOwnerPass)
        {
            // 128 bit RC4
            Console.WriteLine("128 bit RC4暗号化が選択されました。");
            using (PtlEncryptStandard128RC4 enc128r = new PtlEncryptStandard128RC4())
            {
                setModifyWithDefaultPermission(doc, outOwnerPass, enc128r);
            }
        }

        public static void encrypt128AESsetModify(PtlPDFDocument doc, String outOwnerPass)
        {
            // 128 bit AES
            Console.WriteLine("128 bit AES暗号化が選択されました。");
            using (PtlEncryptStandard128AES enc128a = new PtlEncryptStandard128AES())
            {
                setModifyWithDefaultPermission(doc, outOwnerPass, enc128a);
            }
        }

        public static void encrypt256AESsetModify(PtlPDFDocument doc, String outOwnerPass)
        {
            // 256 bit AES
            Console.WriteLine("256 bit AES暗号化が選択されました。");
            using (PtlEncryptStandard256AES enc256 = new PtlEncryptStandard256AES())
            {
                setModifyWithDefaultPermission(doc, outOwnerPass, enc256);
            }
        }
        public static void setModifyWithDefaultPermission(PtlPDFDocument doc, String outOwnerPass,
                                                          PtlEncryptStandard enc)
        {

            // 暗号化するコンテンツ
            // 文書の全てのコンテンツを暗号化します。
            enc.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL);
            // オーナーパスワード値の設定
            enc.setOwnerPassword(outOwnerPass);
            
            using (PtlEncryptPermissionType2 perms2 = new PtlEncryptPermissionType2())
            {
                // 変更権限の設定
                int permModify2 = setModifyForPermType2();
                perms2.setModify((PtlEncryptPermissionType2.PERMISSION_MODIFY)Enum.ToObject(typeof(PtlEncryptPermissionType2.PERMISSION_MODIFY), permModify2));

                // 印刷権限の設定 PERM_PRINT_HIGH : 高解像度での印刷を許可
                perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_HIGH);
                // テキスト、画像、その他の内容のコピーを有効にするかどうかの設定 true: 有効にする
                perms2.setCopy(true);
                // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの設定
                // true: 有効にする
                perms2.setAccessibility(true);

                // ユーザアクセス許可フラグの設定
                enc.setPermission(perms2);
            }

            // 暗号化情報の設定
            doc.setEncrypt(enc);

        }

        public static void setPermModify(PtlPDFDocument doc, String outOwnerPass)
        {
            using (PtlEncrypt encrypt = doc.getEncrypt())
            {

                // 標準セキュリティハンドラのメソッド取得
                if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_STANDARD)
                {
                    // getFilterType()に判定済みなので PtlEncryptStandard にダウンキャストする。
                    PtlEncryptStandard encryptStandard = (PtlEncryptStandard)encrypt;
                    encryptStandard.setOwnerPassword(outOwnerPass);
                    // ユーザアクセス許可フラグの取得
                    using (PtlEncryptPermission perms = encryptStandard.getPermission())
                    {
                        // タイプ取得
                        switch (perms.getType())
                        {
                            case PtlEncryptPermission.PERM_TYPE.PERM_TYPE1:
                                // Type 1は40bit RC4なので、128bit RC4で新規に権限を設定する。
                                Console.WriteLine("入力ファイルは40 bit RC4暗号で暗号化されています。");
                                Console.WriteLine("新規に暗号化します。");
                                encrypt128RC4setModify(doc, outOwnerPass);
                                break;

                            case PtlEncryptPermission.PERM_TYPE.PERM_TYPE2:
                                // getType()で判定されたのでpermsをダウンキャストできる。
                                PtlEncryptPermissionType2 perms2 = (PtlEncryptPermissionType2)perms;
                                // 編集権限設定
                                int permModify2 = setModifyForPermType2();
                                perms2.setModify((PtlEncryptPermissionType2.PERMISSION_MODIFY)Enum.ToObject(typeof(PtlEncryptPermissionType2.PERMISSION_MODIFY), permModify2));
                                encryptStandard.setPermission(perms2);
                                break;
                        }
                    }
                    // 暗号化情報の設定
                    doc.setEncrypt(encryptStandard);
                }
                else
                {
                    Console.WriteLine("標準セキュリティハンドラ以外のセキュリティハンドラが使われたPDFです。");
                    Console.WriteLine("このPDFの変更権限を変更することはできません。");
                    return;
                }
            }
        }

        public static int setModifyForPermType2()
        {
            Console.WriteLine("変更権限の設定をします。以下から数値で指定してください。");
            Console.WriteLine("これ以外の数値が指定された場合は 0 とみなされます。");
            Console.WriteLine("0 : 許可しない");
            Console.WriteLine("1 : 注釈の作成、フォームフィールドの入力と既存の署名フィールドに署名を許可");
            Console.WriteLine("2 : ページの挿入、削除、回転を許可");
            Console.WriteLine("3 : フォームフィールドの入力と既存の署名フィールドに署名を許可");
            Console.WriteLine("4 : ページ抽出を除く全ての動作を許可");

            int choiceOfPerm = int.Parse(Console.ReadLine());

            int permModify2 = 0;
            switch (choiceOfPerm)
            {
                case 0:
                    // PERM_MODIFY_NOT_ALLOWEDを用いるため、変更無し
                    break;
                case 1:
                    permModify2 = 3;
                    break;
                case 2:
                    permModify2 = 1;
                    break;
                case 3:
                    permModify2 = 2;
                    break;
                case 4:
                    permModify2 = 4;
                    break;
                default:
                    // PERM_MODIFY_NOT_ALLOWEDを用いるため、変更無し
                    break;
            }

            return permModify2;
        }
    }
}
