﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：印刷不可セキュリティの設定
  　　　印刷の可否について設定します。
  　　　暗号化済みのファイルの場合はその他のセキュリティ情報は保持されます。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class EncryptSetPrint
    {
        // そのクラスのusageを表示する関数
        public static void printUsage()
        {
            Console.Write("usage: EncryptSetPrint in-pdf-file out-pdf-file ");
            Console.WriteLine("印刷権限種類 out-pdf-ownerpass [in-pdf-password]");
            Console.WriteLine("印刷権限種類\n0 : 印刷不可  1 : 低解像度で可  2 : 高解像度で可");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 4)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の取得
            int permPrint = -1;
            try
            {
                permPrint = int.Parse(args[2]);
                switch (permPrint)
                {
                    case 0:
                    case 1:
                    case 2:
                        break;
                    default:
                        Console.WriteLine("印刷権限種類には、0から2の数値を指定してください。");
                        printUsage(); // usageメッセージの表示
                        return;
                }
            }
            catch (FormatException)
            {
                Console.WriteLine("印刷権限種類には、0から2の数値を指定してください。");
                printUsage(); // usageメッセージの表示
            }
            String outOwnerPass = args[3];


            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    if (args.Length > 4)
                    {
                        String password = (String)args[4];
                        // パスワードのセット
                        doc.setPassword(password);
                    }

                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // 暗号化の有無判定
                    if (doc.isEncrypted())
                    {
                        // 暗号化情報の取得
                        setPermPrint(doc, outOwnerPass, permPrint);
                    }
                    else
                    {
                        Console.WriteLine("暗号化されていないファイルです。\n");
                        Console.WriteLine("印刷権限を設定して暗号化します。");
                        Console.WriteLine("暗号化種類の設定をします。1から3の数値を指定してください。");
                        Console.WriteLine("それ以外の数値が設定された場合は 3 が選択されたものとして扱います。");
                        Console.WriteLine("暗号化種類");
                        Console.WriteLine("1 : 128 bit RC4  2 : 128 bit AES  3 : 256 bit AES");
                        int kind = int.Parse(Console.ReadLine());
                        // セキュリティ設定
                        switch (kind)
                        {
                            case 1:
                                // 128 bit RC4
                                encrypt128RC4setPrint(doc, outOwnerPass, permPrint);
                                break;
                            case 2:
                                // 128 bit AES
                                encrypt128AESsetPrint(doc, outOwnerPass, permPrint);
                                break;
                            case 3:
                            default:
                                // 256 bit AES
                                encrypt256AESsetPrint(doc, outOwnerPass, permPrint);
                                break;
                        }

                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void encrypt128RC4setPrint(PtlPDFDocument doc, String outOwnerPass, int permPrint)
        {
            // 128 bit RC4
            Console.WriteLine("128 bit RC4暗号化が選択されました。");
            using (PtlEncryptStandard128RC4 enc128r = new PtlEncryptStandard128RC4())
            {
                setPrintWithDefaultPermission(doc, outOwnerPass, permPrint, enc128r);
            }
        }

        public static void encrypt128AESsetPrint(PtlPDFDocument doc, String outOwnerPass, int permPrint)
        {
            // 128 bit AES
            Console.WriteLine("128 bit AES暗号化が選択されました。");
            using (PtlEncryptStandard128AES enc128a = new PtlEncryptStandard128AES())
            {
                setPrintWithDefaultPermission(doc, outOwnerPass, permPrint, enc128a);
            }
        }

        public static void encrypt256AESsetPrint(PtlPDFDocument doc, String outOwnerPass, int permPrint)
        {
            // 256 bit AES
            Console.WriteLine("256 bit AES暗号化が選択されました。");
            using (PtlEncryptStandard256AES enc256 = new PtlEncryptStandard256AES())
            {
                setPrintWithDefaultPermission(doc, outOwnerPass, permPrint, enc256);
            }
        }

        public static void setPrintWithDefaultPermission(PtlPDFDocument doc, String outOwnerPass,
                                                         int permPrint, PtlEncryptStandard enc)
        {

            // 暗号化するコンテンツ
            // 文書の全てのコンテンツを暗号化します。
            enc.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL);
            // オーナーパスワード値の設定
            enc.setOwnerPassword(outOwnerPass);
            
            using (PtlEncryptPermissionType2 perms2 = new PtlEncryptPermissionType2())
            {
                // 印刷権限の設定
                switch (permPrint)
                {
                    case 0:
                    default:
                        perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_NOT_ALLOWED);
                        break;
                    case 1:
                        perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_LOW);
                        break;
                    case 2:
                        perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_HIGH);
                        break;
                }

                // 変更権限の設定
                // ページ抽出を除く全ての動作を許可
                perms2.setModify(PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_MODYFYDOC);

                // テキスト、画像、その他の内容のコピーを有効にするかどうかの設定
                // true: 有効にする
                perms2.setCopy(true);

                // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの設定
                // true: 有効にする
                perms2.setAccessibility(true);

                // ユーザアクセス許可フラグの設定
                enc.setPermission(perms2);
            }

            // 暗号化情報の設定
            doc.setEncrypt(enc);
        }

        public static void setPermPrint(PtlPDFDocument doc, String outOwnerPass, int permPrint)
        {
            using (PtlEncrypt encrypt = doc.getEncrypt())
            {

                // 標準セキュリティハンドラのメソッド取得
                if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_STANDARD)
                {
                    // getFilterType()に判定済みなので PtlEncryptStandard にダウンキャストする。
                    PtlEncryptStandard encryptStandard = (PtlEncryptStandard)encrypt;
                    encryptStandard.setOwnerPassword(outOwnerPass);
                    // ユーザアクセス許可フラグの取得
                    using (PtlEncryptPermission perms = encryptStandard.getPermission())
                    {
                        // タイプ取得
                        switch (perms.getType())
                        {
                            case PtlEncryptPermission.PERM_TYPE.PERM_TYPE1:
                                // Type 1は40bit RC4なので、128bit RC4で新規に権限を設定する。
                                Console.WriteLine("入力ファイルは40 bit RC4暗号で暗号化されています。");
                                Console.WriteLine("新規に暗号化します。");
                                encrypt128RC4setPrint(doc, outOwnerPass, permPrint);
                                break;

                            case PtlEncryptPermission.PERM_TYPE.PERM_TYPE2:
                                // getType()で判定されたのでpermsをダウンキャストできる。
                                PtlEncryptPermissionType2 perms2 = (PtlEncryptPermissionType2)perms;
                                switch (permPrint)
                                {
                                    case 0:
                                    default:
                                        perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_NOT_ALLOWED);
                                        break;
                                    case 1:
                                        perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_LOW);
                                        break;
                                    case 2:
                                        perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_HIGH);
                                        break;
                                }
                                encryptStandard.setPermission(perms2);
                                break;
                        }
                    }
                    // 暗号化情報の設定
                    doc.setEncrypt(encryptStandard);
                }
                else
                {
                    Console.WriteLine("標準セキュリティハンドラ以外のセキュリティハンドラが使われたPDFです。");
                    Console.WriteLine("このPDFの印刷権限を変更することはできません。");
                    return;
                }
            }
        }
    }
}
