﻿/*
 Antenna House PDF Tool API V6.0
 .Net Interface sample program

 概要：セキュリティ情報の設定・オーナーパスワードのみ設定する。

 Copyright (C) 2018- Antenna House, Inc. All rights reserved.
 Antenna House is a trademark of [Antenna House, Inc.]

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class EncryptWithOwnerPass
    {
        // そのクラスのusageを表示する関数
        public static void printUsage()
        {
            Console.Write("usage: EncryptWithOwnerPass in-pdf-file out-pdf-file");
            Console.WriteLine("暗号化種類 out-pdf-ownerpass [in-pdf-password]\n");
            Console.WriteLine("暗号化種類");
            Console.WriteLine("1 : 128 bit RC4  2 : 128 bit AES  3 : 256 bit AES");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 4)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            int kind;
            try
            {
                kind = int.Parse(args[2]);
                switch (kind)
                {
                    case 1:
                    case 2:
                    case 3:
                        break;
                    default:
                        Console.WriteLine("暗号化種類には、1から3の数値を指定してください。");
                        printUsage(); // usageメッセージの表示
                        return;
                }
            }
            catch (FormatException)
            {
                Console.WriteLine("暗号化種類には、1から3の数値を指定してください。");
                printUsage(); // usageメッセージの表示
                return;
            }
            // コマンドライン引数の取得
            String outOwnerpass = args[3];
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    if (args.Length > 4)
                    {
                        String password = (String)args[4];
                        // パスワードのセット
                        doc.setPassword(password);
                    }

                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // セキュリティ設定
                    switch (kind)
                    {
                        case 1:
                            // 128 bit RC4
                            encrypt128RC4withOwnerpass(doc, outOwnerpass);
                            break;
                        case 2:
                            // 128 bit AES
                            encrypt128AESwithOwnerpass(doc, outOwnerpass);
                            break;
                        case 3:
                            // 256 bit AES
                            encrypt256AESwithOwnerpass(doc, outOwnerpass);
                            break;
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void encrypt128RC4withOwnerpass(PtlPDFDocument doc, String outOwnerpass)
        {
            // 128 bit RC4
            Console.WriteLine("128 bit RC4暗号化の設定をします。");
            using (PtlEncryptStandard128RC4 enc128r = new PtlEncryptStandard128RC4())
            {
                setEncryptWithOwnerpass(doc, outOwnerpass, enc128r);
            }
        }

        public static void encrypt128AESwithOwnerpass(PtlPDFDocument doc, String outOwnerpass)
        {
            // 128 bit AES
            Console.WriteLine("128 bit AES暗号化の設定をします。");
            using (PtlEncryptStandard128AES enc128a = new PtlEncryptStandard128AES())
            {
                setEncryptWithOwnerpass(doc, outOwnerpass, enc128a);
            }
        }

        public static void encrypt256AESwithOwnerpass(PtlPDFDocument doc, String outOwnerpass)
        {
            // 256 bit AES
            Console.WriteLine("256 bit AES暗号化の設定をします。");
            using (PtlEncryptStandard256AES enc256 = new PtlEncryptStandard256AES())
            {
                setEncryptWithOwnerpass(doc, outOwnerpass, enc256);
            }
        }

        public static void setEncryptWithOwnerpass(PtlPDFDocument doc, String outOwnerpass, PtlEncryptStandard enc)
        {
            // 暗号化するコンテンツを設定
            // ENCRYPT_ALL : 全て暗号化
            enc.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL);
            // ユーザパスワード値の設定
            enc.setOwnerPassword(outOwnerpass);

            // ユーザアクセス許可フラグの取得・設定
            using (PtlEncryptPermissionType2 perms2 = new PtlEncryptPermissionType2())
            {
                // 印刷権限の設定
                // PERM_PRINT_NOT_ALLOWED : 許可しない
                perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_NOT_ALLOWED);

                // 変更権限の設定
                // PERM_MODIFY_NOT_ALLOWED : 許可しない
                perms2.setModify(PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_NOT_ALLOWED);

                // テキスト、画像、その他の内容のコピーを有効にするかどうかの設定
                // false : 有効にしない
                perms2.setCopy(false);

                // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの設定
                // false : 有効にしない
                perms2.setAccessibility(false);

                // ユーザアクセス許可フラグの設定
                enc.setPermission(perms2);
            }
            doc.setEncrypt(enc);
        }
    }
}
