﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：セキュリティ情報の取得

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class GetEncryptInfo
    {
        public static void Main(String[] args)
        {
            if (args.Length < 1)
            {
                Console.WriteLine("usage: GetEncryptInfo in-pdf-file [in-pdf-password]");
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    if (args.Length > 1)
                    {
                        // パスワードのセット
                        doc.setPassword(args[1]);
                    }

                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // 暗号化の取得
                    if (doc.isEncrypted())
                    {
                        // 暗号化情報の取得
                        showEncrypt(doc);
                    }
                    else
                    {
                        Console.WriteLine("暗号化されていないファイルです");
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void showEncrypt(PtlPDFDocument doc)
        {
            using (PtlEncrypt encrypt = doc.getEncrypt())
            {
                // キー長の取得
                Console.WriteLine("暗号化のキー長 : " + encrypt.getKeyLength());

                // 暗号化する文書コンポーネント取得
                switch (encrypt.getEncryptComponent())
                {
                    case PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL:
                        Console.WriteLine("文書の全てのコンテンツを暗号化");
                        break;
                    case PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_EXCEPT_METADATA:
                        Console.WriteLine("文書のメタデータを除く全てのコンテンツを暗号化");
                        break;
                    case PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ONLY_ATTACHED_FILE:
                        Console.WriteLine("添付ファイルのみを暗号化(AES暗号化のみで有効)");
                        break;
                }

                // 暗号化のフィルター取得
                // FILTER_STANDARD = 1 /* 標準セキュリティハンドラ */
                // Ver5.0 はStandardしかない
                if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_STANDARD)
                {
                    // getFilterType()に判定済みなので PtlEncryptStandard にダウンキャストする。
                    PtlEncryptStandard encryptStandard = (PtlEncryptStandard)encrypt;

                    // 標準セキュリティハンドラのメソッド取得
                    switch (encryptStandard.getMethod())
                    {
                        case PtlEncryptStandard.STANDARD_METHOD_TYPE.METHOD_RC4:
                            Console.WriteLine("標準セキュリティハンドラのメソッド : RC4");
                            break;
                        case PtlEncryptStandard.STANDARD_METHOD_TYPE.METHOD_AES:
                            Console.WriteLine("標準セキュリティハンドラのメソッド : AES");
                            break;
                    }

                    // ユーザアクセス許可フラグの取得
                    using (PtlEncryptPermission perms = encryptStandard.getPermission())
                    {
                        // タイプ取得
                        switch (perms.getType())
                        {
                            case PtlEncryptPermission.PERM_TYPE.PERM_TYPE1:
                                {
                                    Console.WriteLine("標準セキュリティハンドラのリビジョン : 2");
                                    // getType()で判定されたのでpermsをダウンキャストできる。
                                    PtlEncryptPermissionType1 perms1 = (PtlEncryptPermissionType1)perms;
                                    // 印刷権限取得
                                    if (perms1.getPrint())
                                    {
                                        Console.WriteLine("印刷を許可する");
                                    }
                                    else
                                    {
                                        Console.WriteLine("印刷を許可しない");
                                    }
                                    // 変更権限取得
                                    switch (perms1.getModify())
                                    {
                                        case PtlEncryptPermissionType1.PERMISSION_MODIFY.PERM_MODIFY_NOT_ALLOWED:
                                            Console.WriteLine("変更権限 : 許可しない");
                                            break;
                                        case PtlEncryptPermissionType1.PERMISSION_MODIFY.PERM_MODIFY_ANNOTANDFORM:
                                            Console.WriteLine("変更権限 : 注釈の作成、フォームフィールドの入力と既存の署名フィールドに署名");
                                            break;
                                        case PtlEncryptPermissionType1.PERMISSION_MODIFY.PERM_MODIFY_ASSEMBLEANDFORM:
                                            Console.WriteLine("変更権限 : ページレイアウト、フォームフィールドの入力と既存の署名フィールドに署名");
                                            break;
                                        case PtlEncryptPermissionType1.PERMISSION_MODIFY.PERM_MODIFY_MODYFYDOC:
                                            Console.WriteLine("変更権限 : ページ抽出を除く全ての動作");
                                            break;
                                    }
                                    // テキスト、画像、その他の内容のコピーおよびアクセシビリティを有効にするかどうかの取得
                                    if (perms1.getCopy())
                                    {
                                        Console.WriteLine("テキスト、画像、その他の内容のコピーおよびアクセシビリティを有効にする");
                                    }
                                    else
                                    {
                                        Console.WriteLine("テキスト、画像、その他の内容のコピーおよびアクセシビリティを有効にしない");
                                    }
                                }
                                break;

                            case PtlEncryptPermission.PERM_TYPE.PERM_TYPE2:
                                {
                                    Console.WriteLine("標準セキュリティハンドラのリビジョン : 3または4");
                                    // getType()で判定されたのでpermsをダウンキャストできる。
                                    PtlEncryptPermissionType2 perms2 = (PtlEncryptPermissionType2)perms;
                                    // 印刷権限取得
                                    switch (perms2.getPrint())
                                    {
                                        case PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_NOT_ALLOWED:
                                            Console.WriteLine("印刷権限 : 許可しない");
                                            break;
                                        case PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_LOW:
                                            Console.WriteLine("印刷権限 : 低解像度");
                                            break;
                                        case PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_HIGH:
                                            Console.WriteLine("印刷権限 : 高解像度");
                                            break;
                                    }
                                    // 変更権限取得
                                    switch (perms2.getModify())
                                    {
                                        case PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_NOT_ALLOWED:
                                            Console.WriteLine("変更権限 : 許可しない");
                                            break;
                                        case PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_ASSEMBLEDOC:
                                            Console.WriteLine("変更権限 : ページの挿入、削除、回転");
                                            break;
                                        case PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_FILLFORM:
                                            Console.WriteLine("変更権限 : フォームフィールドの入力と既存の署名フィールドに署名");
                                            break;
                                        case PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_ANNOTANDFORM:
                                            Console.WriteLine("変更権限 : 注釈の作成、フォームフィールドの入力と既存の署名フィールドに署名");
                                            break;
                                        case PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_MODYFYDOC:
                                            Console.WriteLine("変更権限 : ページ抽出を除く全ての動作");
                                            break;
                                    }

                                    // テキスト、画像、その他の内容のコピーを有効にするかどうかの取得
                                    if (perms2.getCopy())
                                    {
                                        Console.WriteLine("テキスト、画像、その他の内容のコピーを有効にする");
                                    }
                                    else
                                    {
                                        Console.WriteLine("テキスト、画像、その他の内容のコピーを有効にしない");
                                    }
                                    // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの取得
                                    if (perms2.getAccessibility())
                                    {
                                        Console.WriteLine("スクリーンリーダーデバイスのテキストアクセスを有効にする");
                                    }
                                    else
                                    {
                                        Console.WriteLine("スクリーンリーダーデバイスのテキストアクセスを有効にしない");
                                    }
                                }
                                break;
                        }
                    }
                }
            }
        }
    }
}
