﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：しおり情報の取得
  　　　
  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class GetOutlineInfo
    {
        // そのクラスのusageを表示する関数
        public static void printUsage()
        {
            Console.WriteLine("usage: GetOutline in-pdf-file 取得方法\n");
            Console.WriteLine("取得方法\n0 : １ツリー   1 : 詳細");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 2)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の処理
            bool showDetail;
            try
            {
                showDetail = readBoolArgs(args[1], "取得方法は0か1で指定してください。");
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }

            // しおり情報の取得
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロード
                    doc.load(inputFile);

                    if (!doc.hasOutlines())
                    {
                        Console.WriteLine("このPDFにしおり情報はありません。");
                        return;
                    }

                    using (PtlOutline rootOutline = doc.getRootOutline())
                    {
                        if (rootOutline.hasChild())
                        {
                            using (PtlOutline outline = rootOutline.getFirstChild())
                            {
                                readOutline(outline, 0, showDetail);
                            }
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void readOutline(PtlOutline item, int indent, bool showDetail)
        {
            if (showDetail)
            {
                Console.WriteLine("Title = " + item.getTitle());
                Console.WriteLine("Indent = " + indent);
                if (item.isOpen())
                {
                    Console.WriteLine("PDF表示時の子アウトラインをオープンする");
                }
                else
                {
                    Console.WriteLine("PDF表示時の子アウトラインをオープンしない");
                }
                int flags = (int)item.getFlags();
                if ((flags & (int)PtlOutline.ITEM_FLAG.FLAG_BOLD) == (int)PtlOutline.ITEM_FLAG.FLAG_BOLD)
                    Console.WriteLine("FLAG_BOLD on");
                else
                    Console.WriteLine("FLAG_BOLD off");
                if ((flags & (int)PtlOutline.ITEM_FLAG.FLAG_ITALIC) == (int)PtlOutline.ITEM_FLAG.FLAG_ITALIC)
                    Console.WriteLine("FLAG_ITALIC on");
                else
                    Console.WriteLine("FLAG_ITALIC off");
                using (PtlColorDeviceRGB color = item.getColor())
                {
                    int rr = (int)(color.getR() * 255);
                    int gg = (int)(color.getG() * 255);
                    int bb = (int)(color.getB() * 255);
                    Console.Write("Color = RGB({0:d},{0:d},{0:d})\n", rr, gg, bb);
                }
                using (PtlAction act = item.getAction())
                {
                    showAction(act);
                }
            }
            else
            {
                String indentString = "";
                for (int i = 0; i < indent; ++i)
                {
                    indentString += "  ";
                }
                if (showDetail == false)
                {
                    Console.Write(indentString);
                }
                Console.WriteLine(item.getTitle());
            }

            if (item.hasChild())
            {
                using (PtlOutline firstItem = item.getFirstChild())
                {
                    int indentChild = indent + 1;
                    readOutline(firstItem, indentChild, showDetail);
                }
            }
            if (item.hasNextSibling())
            {
                using (PtlOutline nextItem = item.getNextSibling())
                {
                    readOutline(nextItem, indent, showDetail);
                }
            }
        }

        public static void showAction(PtlAction action)
        {
            switch (action.getType())
            {
                case PtlAction.ACTION_TYPE.TYPE_NONE:
                    {
                        Console.WriteLine("Action = " + PtlAction.ACTION_TYPE.TYPE_NONE +
                                           " : アクションなし");
                        break;
                    }
                case PtlAction.ACTION_TYPE.TYPE_GOTO:
                    {
                        Console.WriteLine("Action = " + PtlAction.ACTION_TYPE.TYPE_GOTO +
                                           " : GoToアクション");
                        PtlActionGoTo act = (PtlActionGoTo)action;
                        // 宛先の取得
                        using (PtlDest dest = act.getDest())
                        {
                            showDest(dest);
                        }
                        break;
                    }
                case PtlAction.ACTION_TYPE.TYPE_GOTO_R:
                    {
                        Console.WriteLine("Action = " + PtlAction.ACTION_TYPE.TYPE_GOTO_R +
                                           " : GoToRアクション");
                        PtlActionGoToR act = (PtlActionGoToR)action;
                        // ファイル間移動用PDFファイル名を取得 getFileName()
                        Console.WriteLine("  FileName = " + act.getFileName());
                        // 新ウィンドウフラグを取得
                        if (act.getNewWindowFlag())
                        {
                            Console.WriteLine("  NewWindowFlag = true: 新ウィンドウでオープンする");
                        }
                        else
                        {
                            Console.WriteLine("  NewWindowFlag = false: しない");
                        }
                        // 宛先の取得
                        using (PtlDest dest = act.getDest())
                        {
                            showDest(dest);
                        }
                        break;
                    }
                case PtlAction.ACTION_TYPE.TYPE_LAUNCH:
                    {
                        Console.WriteLine("Action = " + PtlAction.ACTION_TYPE.TYPE_LAUNCH +
                                           " : Launchアクション");
                        PtlActionLaunch act = (PtlActionLaunch)action;
                        // 起動ファイル名を取得
                        Console.WriteLine("  FileName = " + act.getFileName());
                        // 新ウィンドウフラグを取得
                        if (act.getNewWindowFlag())
                        {
                            Console.WriteLine("  NewWindowFlag = true: 新ウィンドウでオープンする");
                        }
                        else
                        {
                            Console.WriteLine("  NewWindowFlag = false: しない");
                        }
                        break;
                    }
                case PtlAction.ACTION_TYPE.TYPE_URI:
                    {
                        Console.WriteLine("Action = " + PtlAction.ACTION_TYPE.TYPE_URI +
                                           " : URIアクション");
                        PtlActionURI act = (PtlActionURI)action;
                        // URIを取得
                        Console.WriteLine("  URI = " + act.getURI());
                        break;
                    }
                case PtlAction.ACTION_TYPE.TYPE_UNKNOWN:
                    {
                        Console.WriteLine("Action = " + PtlAction.ACTION_TYPE.TYPE_UNKNOWN +
                                           " : 未対応アクション");
                        break;
                    }
            }
        }


        public static void showDest(PtlDest dest)
        {
            switch (dest.getType())
            {
                case PtlDest.DEST_TYPE.TYPE_NONE:
                    Console.WriteLine("Dest = " + PtlDest.DEST_TYPE.TYPE_NONE + " : 宛先なし");
                    break;
                case PtlDest.DEST_TYPE.TYPE_XYZ:
                    Console.WriteLine("Dest = " + PtlDest.DEST_TYPE.TYPE_XYZ + " : XYZ型");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT:
                    Console.WriteLine("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT + " : Fit型(全体表示)");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_H:
                    Console.WriteLine("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_H +
                                       " : FitH型(幅に合わせる)");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_V:
                    Console.WriteLine("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_V +
                                       " : FitV型(高さに合わせる)");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_R:
                    Console.WriteLine("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_R + " : FitR型");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_B:
                    Console.WriteLine("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_B + " : FitB型");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_BH:
                    Console.WriteLine("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_BH +
                                       " : FitBH型(描画領域の幅に合わせる)");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_BV:
                    Console.WriteLine("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_BV + " : FitBV型");
                    break;
            }
            // 宛先ページの取得
            Console.WriteLine("宛先ページ(PageNumber) : " + (dest.getPageNumber()) + " ページ目");
        }


        /**
         * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
         *
         * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
         * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
         * @return argsの数値を読み取った結果を戻す
         * @throws ArgumentException argsが0か1でなかった場合に発生。
         */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;

            // argsの読み取り
            switch (args)
            {
                case "0":
                    trueOrFalse = false;
                    break;
                case "1":
                    trueOrFalse = true;
                    break;
                default:
                    throw new ArgumentException(errorMessage);
            }

            return trueOrFalse;
        }
    }
}
