﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：PDF 文書の指定したページにPDF透かしを挿入します。
  　　　指定したPDFの指定したページを読み取ってそれを透かしとします。
  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class PdfWaterMarkAddDecoString
    {
        // そのクラスのusageを表示する関数
        public static void printUsage()
        {
            Console.Write("usage: PdfWaterMarkAddDecoString in-pdf-file out-pdf-file");
            Console.WriteLine(" string-to-deco");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の読み取り
            String stringToDeco = args[2];
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);
                    // 透かしの追加
                    appendPdfWaterMark(doc, stringToDeco);
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void appendPdfWaterMark(PtlPDFDocument doc, String stringToDeco)
        {
            using (PtlParamWaterMarkPDF waterMarkPdf = new PtlParamWaterMarkPDF())
            {
                // 透かしに使用するページの作成・取得
                using (PtlPDFDocument doc_watermark = new PtlPDFDocument())
                using (PtlPages pages = doc_watermark.getPages()) // ページコンテナの取得
                using (PtlPage blancPage = new PtlPage()) // 作成予定のページオブジェクトを作成
                {
                    pages.append(blancPage, PtlPages.INSERT_OPTION.OPTION_NONE); // ページの追加
                    using (PtlPage page = pages.get(0))  // 追加したページを指定
                    {
                        // 追加したページに文字・枠を配置
                        WriteDecoratedString(page, stringToDeco);
                        // 透かしに使用するPDF文書ページを設定
                        waterMarkPdf.setPage(page);
                    }
                }
                // 透かしの倍率の設定
                waterMarkPdf.setScale(1.0f);


                // 以下、PDF透かし以外と共通するデフォルトの設定
                // 透かしの名前の設定
                waterMarkPdf.setName("waterMarkPdf");
                // 透かしを配置するときの余白の設定
                waterMarkPdf.setMargin(10.0f, 10.0f, 10.0f, 10.0f);
                // 透かしの配置の設定
                waterMarkPdf.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
                // 透かしのZオーダーの設定 ZORDER_FRONT = 1
                waterMarkPdf.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
                // 透かしの不透明度の設定
                waterMarkPdf.setOpacity(0.5f);
                // 透かしをタイリングして配置するかどうかの設定
                // false : タイリングして配置しない
                waterMarkPdf.setTiling(false);
                // デフォルトの設定終わり

                // 透かしを文書に設定する
                doc.appendWaterMark(waterMarkPdf);
            }
        }


        // stringToDecoに枠を作成する
        // 最初にstringToDecoに基づいて文字を配置し、矩形を後付けで作成する。
        public static void WriteDecoratedString(PtlPage page, String stringToDeco)
        {
            using (PtlContent content = page.getContent())
            {
                // 文字の描画に使うパラメータクラス
                using (PtlParamWriteString writeString = new PtlParamWriteString())
                using (PtlParamFont font = new PtlParamFont()) //フォント指定に使うパラメータクラス
                using (PtlParamDrawShape drawShape = new PtlParamDrawShape())
                {
                    // フォントサイズ：24.0f = 24pt
                    float fontSize = 24.0f;
                    // フォント名の設定
                    font.setName("ＭＳ Ｐゴシック");
                    // フォントサイズの設定 : 24.0f = 24pt
                    font.setSize(fontSize);
                    // フォントの設定
                    writeString.setFont(font);
                    // 文字色設定
                    writeString.setTextColor(new PtlColorDeviceRGB(0.0f, 1.0f, 0.0f));

                    // 挿入する文字列を囲む矩形のサイズ決定
                    float textH = (25.4f * fontSize) / 72.0f;   //文字列の高さ
                    float textW = font.getStringWidth(stringToDeco);    //文字列の幅
                    PtlRect rectTxt = new PtlRect(2, 2, textW + 2, textH + 2);
                    PtlRect rectPage = new PtlRect(0, 0, textW + 4, textH + 4);

                    // テキスト枠を描画する
                    // 枠の色設定
                    drawShape.setLineColor(new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f));
                    drawShape.setFillColor(new PtlColorNone());

                    // 追加する空白ページを作成するために、先頭ページのサイズに合わせます。
                    page.setViewBox(rectPage);

                    // 文字列出力
                    content.writeString(rectTxt, PtlContent.ALIGN.ALIGN_BOTTOM_LEFT,
                                        stringToDeco, writeString);
                    // 枠の出力
                    content.drawRect(rectTxt, drawShape);
                }
            }
        }
    }
}
