﻿/*
 Antenna House PDF Tool API V7.0
 .Net Interface sample program

 概要：閲覧場所の制限設定(URLで指定)

 Copyright (C) 2018- Antenna House, Inc. All rights reserved.
 Antenna House is a trademark of [Antenna House, Inc.]

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class SetRestrictionByURI
    {
        /**
     * @param args the command line arguments
     */
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                Console.Write("usage: SetRestrictionByURI in-pdf-file out-pdf-file ");
                Console.WriteLine("out-pdf-ownerpass [in-pdf-password]");
                Console.WriteLine("in-pdf-fileが暗号化されていない場合、40bitRC4で暗号化されていた場合は128 bit AES で暗号化します。");
                return;
            }

            // コマンドライン引数の取得
            String outOwnerPass = args[2];
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    if (args.Length > 3)
                    {
                        String password = (String)args[3];
                        // パスワードのセット
                        doc.setPassword(password);
                    }

                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // 閲覧場所の制限設定
                    setRestrictionByURI(doc);

                    if (doc.isEncrypted())
                    {
                    }
                    else
                    {
                        Console.WriteLine("暗号化されていないファイルです。");
                        encrypt128AESwithOwnerpass(doc, outOwnerPass);
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void setRestrictionByURI(PtlPDFDocument doc)
        {
            using (PtlParamRestriction restriction = new PtlParamRestriction())
            {
                // 閲覧可能とするURLの追加
                restriction.appendValidURL("C:\\temp");

                // 透かしの設定
                setWaterMarkToRestriction(restriction);

                // 閲覧制限の設定
                doc.setRestriction(restriction);
            }
        }

        public static void setWaterMarkToRestriction(PtlParamRestriction restriction)
        {
            using (PtlParamWaterMarkText watermarktext = new PtlParamWaterMarkText())
            {
                // 透かしの配置の設定 ALIGN_CENTER = 5 /* 中央 */
                watermarktext.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);

                // 透かしに指定する文字列の設定
                watermarktext.setString("Restriction Test");

                // 透かしに指定するフォントの設定
                using (PtlParamFont font = new PtlParamFont("Helvetica", 180.0f,
                                                          PtlParamFont.WEIGHT.WEIGHT_HEAVY,
                                                          true, false))
                {
                    watermarktext.setFont(font);
                }

                // 透かしの文字に指定する色の設定
                using (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f))
                {
                    watermarktext.setTextColor(colorText);
                }

                // 透かしの文字の縁取りに指定する色の設定
                using (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(0.0f, 1.0f, 0.0f))
                {
                    watermarktext.setOutlineColor(colorOutline);
                }

                // 透かしのテキストを対角線上に配置する設定
                watermarktext.setWriteDiagonal(true);

                // 閲覧不可時に表示するウォーターマーク設定
                restriction.setWatermark(watermarktext);
            }
        }


        public static void encrypt128AESwithOwnerpass(PtlPDFDocument doc, String outOwnerPass)
        {
            // 128 bit AES
            Console.WriteLine("128 bit AES形式で暗号化します。");
            using (PtlEncryptStandard128AES enc128a = new PtlEncryptStandard128AES())
            {
                setOwnerPass(doc, outOwnerPass, enc128a);
            }
        }

        public static void setOwnerPass(PtlPDFDocument doc, String outOwnerPass, PtlEncrypt encrypt)
        {

            // 標準セキュリティハンドラのメソッド取得
            if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_STANDARD)
            {
                // getFilterType()に判定済みなので PtlEncryptStandard にダウンキャストする。
                PtlEncryptStandard encryptStandard = (PtlEncryptStandard)encrypt;
                encryptStandard.setOwnerPassword(outOwnerPass);
                // 暗号化情報の設定
                doc.setEncrypt(encryptStandard);
            }
            else
            {
                Console.WriteLine("標準セキュリティハンドラ以外のセキュリティハンドラが使われたPDFです。");
                Console.WriteLine("このPDFの印刷権限を変更することはできません。");
                return;
            }
        }
    }
}
