﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：PDF ページ数をカウントして、フッターにページ番号を付けます。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class TextWaterMarkPageNumFooter
    {
        // そのクラスのusageを表示する関数
        public static void printUsage()
        {
            Console.WriteLine("usage: TextWaterMarkPageNumFooter in-pdf-file out-pdf-file");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 2)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);
                    // 透かしの追加
                    appendPageNumFooter(doc);
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void appendPageNumFooter(PtlPDFDocument doc)
        {
            using (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText())
            {
                // 以下、フッターの書式についてデフォルトの設定をする
                // 透かしの名前の設定
                waterMarkText.setName("pageNumFooter");
                // 透かしのZオーダーの設定 ZORDER_FRONT = 1
                waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
                // 透かしの不透明度の設定
                waterMarkText.setOpacity(1.0f);
                // 透かしをタイリングして配置するかどうかの設定
                // false : タイリングして配置しない
                waterMarkText.setTiling(false);
                // 透かしに指定するフォントの設定
                using (PtlParamFont font = new PtlParamFont("Times New Roman", 18.0f,
                                                          PtlParamFont.WEIGHT.WEIGHT_MEDIUM, false, true))
                {
                    waterMarkText.setFont(font);
                }
                // 透かしの文字に指定する色の設定
                using (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(0.0f, 0.0f, 0.0f))
                {
                    waterMarkText.setTextColor(colorText);
                }
                // 透かしの文字の縁取りに指定する色の設定
                using (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(0.0f, 0.0f, 0.0f))
                {
                    waterMarkText.setOutlineColor(colorOutline);
                }
                // 透かしのテキストを対角線上に配置する設定
                // false : 対角線上に配置しない
                waterMarkText.setWriteDiagonal(false);
                // 透かしのテキストを任意の角度で配置する設定
                waterMarkText.setTextAngle(0.0f);
                // フッターのデフォルト設定終わり



                // マージンを設定する
                waterMarkText.setMargin(5.0f, 5.0f, 10.0f, 5.0f);
                // 透かしの配置の設定 ALIGN_BOTTOM : 中央下
                waterMarkText.setAlign(PtlParamWaterMark.ALIGN.ALIGN_BOTTOM);
                // 透かしを入れるページの範囲の設定 PAGE_RANGE_ALL : ページ範囲指定
                waterMarkText.setPageRange(PtlParamWaterMark.PAGE_RANGE.PAGE_RANGE_CUSTOM);

                int numPages = doc.getPageCount(); // 総ページ数の取得
                Console.WriteLine("ページ数：" + numPages);
                Console.WriteLine("処理中…");
                for (int i = 0; i < numPages; i++)
                {
                    // 透かしを読み込んだページに配置するよう設定
                    waterMarkText.setCustomPageRange(i.ToString());
                    // 透かしに指定する文字列の設定
                    waterMarkText.setString((i + 1) + "/ " + numPages + "pages");

                    // 透かしを文書に設定する
                    doc.appendWaterMark(waterMarkText);
                }
            }
        }
    }
}
