﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：PDF 文書の指定したページにテキストを透かしとして挿入します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class TextWaterMarkSetAngle
    {
        // そのクラスのusageを表示する関数
        public static void printUsage()
        {
            Console.Write("usage: TextWaterMarkSetAngle in-pdf-file out-pdf-file");
            Console.WriteLine(" write-diagonal [text-angle]");
            Console.WriteLine("write-diagonal : ");
            Console.WriteLine("0:透かしのテキストを対角線上に配置しない  1:配置する");
            Console.WriteLine("write-diagonalが1の場合、text-angleを指定しても無視されます。");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の読み取り
            bool writeDiagonal;
            try
            {
                writeDiagonal = readBoolArgs(args[2], "write-diagonalは0か1で指定してください。");
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }
            float textAngle = float.Parse(args[3]);

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);
                    // 透かしの追加
                    appendTextWaterMarkSetAngle(doc, writeDiagonal, textAngle);
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void appendTextWaterMarkSetAngle(PtlPDFDocument doc, bool writeDiagonal,
                                                       float textAngle)
        {
            using (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText())
            {
                // 透かしのテキストを対角線上に配置する設定
                // true: 対角線上に配置する false : 対角線上に配置しない
                waterMarkText.setWriteDiagonal(writeDiagonal);
                // 透かしのテキストを任意の角度で配置する設定
                waterMarkText.setTextAngle(textAngle);


                // 以下、その他の項目について透かしであることが分かりやすいようデフォルトの設定をする
                // 透かしの名前の設定
                waterMarkText.setName("waterMarkText");
                // 透かしの配置の設定
                waterMarkText.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
                // 透かしのZオーダーの設定 ZORDER_FRONT = 1
                waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
                // 透かしの不透明度の設定
                waterMarkText.setOpacity(1.0f);
                // 透かしをタイリングして配置するかどうかの設定
                // false : タイリングして配置しない
                waterMarkText.setTiling(false);
                // 透かしに指定する文字列の設定
                waterMarkText.setString("WaterMarkText test");
                // 透かしに指定するフォントの設定
                using (PtlParamFont font = new PtlParamFont("Times New Roman", 72.0f,
                                                          PtlParamFont.WEIGHT.WEIGHT_MEDIUM,
                                                          false, true))
                {
                    waterMarkText.setFont(font);
                }
                // 透かしの文字に指定する色の設定
                using (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f))
                {
                    waterMarkText.setTextColor(colorText);
                }
                // 透かしの文字の縁取りに指定する色の設定
                using (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(0.0f, 0.0f, 0.0f))
                {
                    waterMarkText.setOutlineColor(colorOutline);
                }
                // デフォルトの設定終わり

                // 透かしを文書に設定する
                doc.appendWaterMark(waterMarkText);
            }
        }

        /**
         * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
         *
         * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
         * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
         * @return argsの数値を読み取った結果を戻す
         * @throws ArgumentException argsが0か1でなかった場合に発生。
         */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;

            // argsの読み取り
            switch (args)
            {
                case "0":
                    trueOrFalse = false;
                    break;
                case "1":
                    trueOrFalse = true;
                    break;
                default:
                    throw new ArgumentException(errorMessage);
            }

            return trueOrFalse;
        }
    }
}
