﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：透かしの挿入矩形・配置を指定する

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class WaterMarkSetRect
    {
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                Console.WriteLine("usage: WaterMarkSetRect in-pdf-file out-pdf-file watermark-align");
                return;
            }

            String waterMarkAlign = args[2];
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // 透かしの追加
                    appendWaterMarkSetRect(doc, waterMarkAlign);

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void appendWaterMarkSetRect(PtlPDFDocument doc, String waterMarkAlign)
        {
            using (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText())
            {

                // 透かしを配置する矩形の設定(setOutputRect()を使って改めて指定する。)
                Console.WriteLine("透かしを配置する矩形の設定を行います。");
                using (PtlRect rect = setOutputRect())
                {
                    waterMarkText.setRect(rect);
                }

                // 透かしの配置の設定
                Console.WriteLine("透かしの配置(watermark-align)を設定します。: " + waterMarkAlign);
                try
                {
                    waterMarkText.setAlign((PtlParamWaterMark.ALIGN)Enum.Parse(typeof(PtlParamWaterMark.ALIGN),waterMarkAlign));
                }
                catch (ArgumentException ex)
                {//PtlContent.ALIGNの指定が誤っていた場合のエラーメッセージ
                    Console.WriteLine(ex.Message);
                    Console.WriteLine("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                    Console.WriteLine(ex.StackTrace);
                }


                // 以下、その他の項目について透かしであることが分かりやすいようデフォルトの設定をする
                // 透かしの名前の設定
                waterMarkText.setName("nameOfWaterMark");
                // 透かしのZオーダーの設定 ZORDER_FRONT = 1
                waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
                // 透かしを入れるページの範囲の設定 PAGE_RANGE_ALL = 0 /* 全ページ */
                waterMarkText.setPageRange(PtlParamWaterMark.PAGE_RANGE.PAGE_RANGE_ALL);
                // 透かしの不透明度の設定
                waterMarkText.setOpacity(1.0f);
                // 透かしをタイリングして配置するかどうかの設定
                // false : タイリングして配置しない
                waterMarkText.setTiling(false);
                // 透かしに指定する文字列の設定
                waterMarkText.setString("WaterMark Test");
                // 透かしに指定するフォントの設定
                using (PtlParamFont font = new PtlParamFont("Times New Roman", 72.0f, PtlParamFont.WEIGHT.WEIGHT_MEDIUM, false, true))
                {
                    waterMarkText.setFont(font);
                }
                // 透かしの文字に指定する色の設定
                using (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f))
                {
                    waterMarkText.setTextColor(colorText);
                }
                // 透かしの文字の縁取りに指定する色の設定
                using (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(0.0f, 1.0f, 0.0f))
                {
                    waterMarkText.setOutlineColor(colorOutline);
                }
                // 透かしのテキストを対角線上に配置する設定
                // false : 対角線上に配置しない
                waterMarkText.setWriteDiagonal(false);
                // 透かしのテキストを任意の角度で配置する設定
                waterMarkText.setTextAngle(15.0f);
                // デフォルトの設定終わり

                // 透かしを文書に設定する
                doc.appendWaterMark(waterMarkText);
            }
        }

        /**
         * 矩形の数値を入力してその数値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         *
         * @param br BufferedReader。数値の読み取りに使う。
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         */
        public static PtlRect setOutputRect()
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            PtlRect outputRect = new PtlRect();
            while (!isValueOkay)
            {
                Console.WriteLine("配置矩形の各数値を入力してください。");
                Console.Write("top (mm) : ");
                top = float.Parse(Console.ReadLine());
                Console.Write("bottom (mm) : ");
                bottom = float.Parse(Console.ReadLine());
                if (top < bottom)
                {
                    Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                    continue;
                }
                Console.Write("left (mm) : ");
                left = float.Parse(Console.ReadLine());
                Console.Write("right (mm) : ");
                right = float.Parse(Console.ReadLine());
                if (right < left)
                {
                    Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                    continue;
                }

                //矩形を正しく指定できた場合の処理
                isValueOkay = true;
                outputRect.setLeft(left);
                outputRect.setBottom(bottom);
                outputRect.setRight(right);
                outputRect.setTop(top);
            }
            return outputRect;
        }
    }
}
