﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：透かしを前面に配置するか、背後に配置するか指定する

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class WaterMarkSetZorder
    {
        // そのクラスのusageを表示する関数
        public static void printUsage()
        {
            Console.WriteLine("usage: WaterMarkSetZorder in-pdf-file out-pdf-file z-order");
            Console.WriteLine("z-order : 透かしのZオーダー\n 0 : 背面, 1 : 前面");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                printUsage(); //usageの表示
                return;
            }

            int zOrder = int.Parse(args[2]);

            //コマンドライン引数の判定
            switch (zOrder)
            {
                case 0:
                case 1:
                    break;
                default:
                    Console.WriteLine("z-orderは0か1の数値で指定してください。");
                    printUsage(); // usageメッセージの表示
                    return;
            }
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);
                    // 透かしの追加
                    appendWaterMarkSetZorder(doc, zOrder);
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void appendWaterMarkSetZorder(PtlPDFDocument doc, int zOrder)
        {
            using (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText())
            {
                // 透かしのZオーダーの設定
                switch (zOrder)
                {
                    case 0:
                        waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_BACK);
                        break;
                    case 1:
                        waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
                        break;
                }


                // 以下、その他の項目について透かしであることが分かりやすいようデフォルトの設定をする
                // 透かしの配置の設定
                waterMarkText.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
                // 透かしの名前の設定
                waterMarkText.setName("nameOfWaterMark");
                // 透かしを入れるページの範囲の設定 PAGE_RANGE_ALL = 0 /* 全ページ */
                waterMarkText.setPageRange(PtlParamWaterMark.PAGE_RANGE.PAGE_RANGE_ALL);
                // 透かしの不透明度の設定
                waterMarkText.setOpacity(1.0f);
                // 透かしをタイリングして配置するかどうかの設定
                // false : タイリングして配置しない
                waterMarkText.setTiling(false);
                // 透かしに指定する文字列の設定
                waterMarkText.setString("WaterMark Test");
                // 透かしに指定するフォントの設定
                using (PtlParamFont font = new PtlParamFont("Times New Roman", 72.0f,
                                                          PtlParamFont.WEIGHT.WEIGHT_MEDIUM, false, true))
                {
                    waterMarkText.setFont(font);
                }
                // 透かしの文字に指定する色の設定
                using (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f))
                {
                    waterMarkText.setTextColor(colorText);
                }
                // 透かしの文字の縁取りに指定する色の設定
                using (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(0.0f, 1.0f, 0.0f))
                {
                    waterMarkText.setOutlineColor(colorOutline);
                }
                // 透かしのテキストを対角線上に配置する設定
                // false : 対角線上に配置しない
                waterMarkText.setWriteDiagonal(false);
                // 透かしのテキストを任意の角度で配置する設定
                waterMarkText.setTextAngle(15.0f);
                // デフォルトの設定終わり

                // 透かしを文書に設定する
                doc.appendWaterMark(waterMarkText);
            }
        }
    }
}
