/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：PDF 文書の指定したページにPDF透かしを挿入します。
  　　　指定したPDFの指定したページを読み取ってそれを透かしとします。
  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class ColorWaterMarkAppend {
    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.print("usage: java ColorWaterMarkAppend in-pdf-file out-pdf-file");
        System.out.println(" colorR colorG colorB\n");
        System.out.println("color-R, color-G, color-B :");
        System.out.println("0.0～1.0の数値で指定。それぞれの色について1.0が最大");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage(); // usageメッセージの表示
            return;
        }
        // コマンドライン引数の読み取り
        float colorR = Float.parseFloat(args[2]);
        float colorG = Float.parseFloat(args[3]);
        float colorB = Float.parseFloat(args[4]);

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            // 透かしの追加
            appendColorWaterMark(doc, colorR, colorG, colorB);
            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void appendColorWaterMark(PtlPDFDocument doc, float colorR,
                                            float colorG, float colorB)
        throws PtlException, Exception, Error {
        try (PtlParamWaterMarkColor waterMarkColor = new PtlParamWaterMarkColor()) {
            //透かしを配置する矩形の設定
            waterMarkColor.setMargin(10.0f, 10.0f, 10.0f, 10.0f);
            //透かしに指定する色の設定
            try (PtlColorDeviceRGB color = new PtlColorDeviceRGB(colorR, colorG, colorB)) {
                waterMarkColor.setColor(color);
            }

            // 以下、PDF透かし以外と共通するデフォルトの設定
            // 透かしの名前の設定
            waterMarkColor.setName("waterMarkColor");
            // 透かしの配置の設定
            waterMarkColor.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
            // 透かしのZオーダーの設定 ZORDER_FRONT = 1 
            waterMarkColor.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
            // 透かしの不透明度の設定
            waterMarkColor.setOpacity(0.5f);
            // デフォルトの設定終わり

            // 透かしを文書に設定する
            doc.appendWaterMark(waterMarkColor);
        }
    }

}
