/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：利用の制限
  　　　内容のコピー関する制限を設定します。
  　　　暗号化済みのファイルの場合はその他のセキュリティ情報は保持されます。
  　　　40 bit RC4 暗号化の場合はコピーの可否とアクセシビリティ有効の可否が連動します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;

public class EncryptSetAccessibility {

    public static void printUsage() {
        System.out.print("usage: java EncryptSetAccessibility in-pdf-file out-pdf-file ");
        System.out.println("アクセシビリティ有効の可否 out-pdf-ownerpass [in-pdf-password]");
        System.out.print("アクセシビリティ（スクリーンリーダーデバイスのテキストアクセス）を");
        System.out.println("有効にするか");
        System.out.println("0 : 有効にしない 1 : 有効にする");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の取得
        boolean permAccessibility = false;
        try {
            switch (Integer.parseInt(args[2])) {
            case 0:
                permAccessibility = false;
                System.out.println("この文書のアクセシビリティ（スクリーンリーダーデバイスのテキストアクセス）を有効にしません。");
                break;
            case 1:
                permAccessibility = true;
                System.out.println("この文書のアクセシビリティ（スクリーンリーダーデバイスのテキストアクセス）を有効にします。");
                break;
            default:
                System.out.println("アクセシビリティ有効の可否は、0か1の数値で指定してください。");
                printUsage(); // usageメッセージの表示
                return;
            }
        }
        catch(NumberFormatException numfe) {
            System.out.println("アクセシビリティ有効の可否は、0か1の数値で指定してください。");
            printUsage(); // usageメッセージの表示
            return;
        }
        String outOwnerPass = args[3];

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            if (args.length > 4) {
                String password = (String)args[4];
                // パスワードのセット
                doc.setPassword(password);
            }

            // PDFファイルをロードします。
            doc.load(inputFile);

            // 暗号化の有無判定
            if (doc.isEncrypted()) {
                // 暗号化情報の取得
                setPermAccessibility(doc, outOwnerPass, permAccessibility);
            }
            else {
                System.out.println("暗号化されていないファイルです。\n");
                System.out.println("アクセシビリティ有効の可否を設定して暗号化します。");
                System.out.println("暗号化種類の設定をします。0から3の数値を指定してください。");
                System.out.println("それ以外の数値が設定された場合は 3 が選択されたものとして扱います。");
                System.out.println("暗号化種類\n0 : 128 bit RC4  1 : 128 bit AES  2 : 256 bit AES");
                int kind;
                try {
                    kind = Integer.parseInt(br.readLine());
                    // セキュリティ設定
                    switch (kind) {
                    case 0:
                        // 128 bit RC4
                        encrypt128RC4setAccessibility(doc, outOwnerPass, permAccessibility);
                        break;
                    case 1:
                        // 128 bit AES
                        encrypt128AESsetAccessibility(doc, outOwnerPass, permAccessibility);
                        break;
                    case 2:
                    default:
                        // 256 bit AES
                        encrypt256AESsetAccessibility(doc, outOwnerPass, permAccessibility);
                        break;
                    }
                }
                catch (NumberFormatException numfe) {
                    System.out.println("暗号化の種類指定は整数を用いてください。");
                    numfe.printStackTrace();
                }
                
            }
            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }


    public static void encrypt128RC4setAccessibility(PtlPDFDocument doc, String outOwnerPass,
                                                     boolean permAccessibility)
        throws PtlException, IOException, Exception, Error {
        // 128 bit RC4
        System.out.println("128 bit RC4暗号化が選択されました。");
        try (PtlEncryptStandard128RC4 enc128r = new PtlEncryptStandard128RC4()) {
            setAccessibilityWithDefaultPermission(doc, outOwnerPass, permAccessibility, enc128r);
        }
    }

    public static void encrypt128AESsetAccessibility(PtlPDFDocument doc, String outOwnerPass,
                                                     boolean permAccessibility)
        throws PtlException, IOException, Exception, Error {
        // 128 bit AES
        System.out.println("128 bit AES暗号化が選択されました。");
        try (PtlEncryptStandard128AES enc128a = new PtlEncryptStandard128AES()) {
            setAccessibilityWithDefaultPermission(doc, outOwnerPass, permAccessibility, enc128a);
        }
    }

    public static void encrypt256AESsetAccessibility(PtlPDFDocument doc, String outOwnerPass,
                                                     boolean permAccessibility)
        throws PtlException, IOException, Exception, Error {
        // 256 bit AES
        System.out.println("256 bit AES暗号化が選択されました。");
        try (PtlEncryptStandard256AES enc256 = new PtlEncryptStandard256AES()) {
            setAccessibilityWithDefaultPermission(doc, outOwnerPass, permAccessibility, enc256);
        }
    }

    public static void setAccessibilityWithDefaultPermission(PtlPDFDocument doc, String outOwnerPass,
                                                             boolean permAccessibility,
                                                             PtlEncryptStandard enc)
        throws PtlException, IOException, Exception, Error {
        // 暗号化するコンテンツ
        // 文書の全てのコンテンツを暗号化します。
        enc.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL);
        // オーナーパスワード値の設定
        enc.setOwnerPassword(outOwnerPass);
        // PtlEncryptPermissionType2で権限設定を行う
        try (PtlEncryptPermissionType2 perms2 = new PtlEncryptPermissionType2()) {
            // 印刷権限の設定
            // PERM_PRINT_HIGH : 高解像度での印刷を許可
            perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_HIGH);

            // 変更権限の設定
            // ページ抽出を除く全ての動作を許可
            perms2.setModify(PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_MODYFYDOC);

            // 内容のコピーを有効にするかどうかの設定
            perms2.setCopy(false);

            // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの設定
            // true: 有効にする
            perms2.setAccessibility(permAccessibility);

            // ユーザアクセス許可フラグの設定
            enc.setPermission(perms2);
        }
        // 暗号化情報の設定
        doc.setEncrypt(enc);

    }

    public static void setPermAccessibility(PtlPDFDocument doc, String outOwnerPass,
                                            boolean permAccessibility)
        throws PtlException, IOException, Exception, Error {
        try (PtlEncrypt encrypt = doc.getEncrypt()) {
            
            
            if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_STANDARD) {
                // getFilterType()に判定済みなので PtlEncryptStandard にダウンキャストする。
                // 標準セキュリティハンドラのメソッド取得
                try (PtlEncryptStandard encryptStandard = (PtlEncryptStandard)encrypt;
                     PtlEncryptPermission perms = encryptStandard.getPermission()) { // ユーザアクセス許可フラグの取得
                    // タイプ取得
                    switch (perms.getType()) {
                    case PERM_TYPE1:
                        // Type 1は40bit RC4なので、128bit RC4で新規に権限を設定する。
                        System.out.println("入力ファイルは40 bit RC4暗号で暗号化されています。");
                        System.out.println("新規に暗号化します。");
                        encrypt128RC4setAccessibility(doc, outOwnerPass, permAccessibility);
                        break;

                    case PERM_TYPE2:
                        // getType()で判定されたのでpermsをダウンキャストできる。
                        try(PtlEncryptPermissionType2 perms2 = (PtlEncryptPermissionType2)perms) {
                        // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの設定
                            if(permAccessibility){
                                //有効にする場合：コピー禁止設定の有無はそのまま引き継ぐ
                                perms2.setAccessibility(permAccessibility);
                                encryptStandard.setPermission(perms2);
                                break;
                            }else {
                                //無効にする場合：コピー禁止の設定を行う
                                perms2.setCopy(false);
                                perms2.setAccessibility(permAccessibility);
                                encryptStandard.setPermission(perms2);
                            }
                        }
                        encryptStandard.setOwnerPassword(outOwnerPass);
                        doc.setEncrypt(encryptStandard);
                        break;
                    }
                }
            }
            else {
                System.out.println("標準セキュリティハンドラ以外のセキュリティハンドラが使われたPDFです。");
                System.out.println("このPDFの権限を変更することはできません。");
                return;
            }
        }
    }

}

