/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：暗号化の対象の設定
  　　　文書の全てのコンテンツを暗号化、
  　　　文書のメタデータを除くすべてのコンテンツを暗号化、
  　　　添付ファイルのみ暗号化の３つから選択します。
  　　　暗号化形式は自動的に256 bit AES 暗号化に変換されます。
  　　　全てのユーザアクセス許可フラグは破棄されます。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;

public class EncryptSetEncComponent {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            System.out.print("usage: java EncryptSetEncComponent in-pdf-file out-pdf-file ");
            System.out.println("out-pdf-userpass [in-pdf-password]");
            return;
        }

        // コマンドライン引数の取得
        String outUserPass = args[2];

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            if (args.length > 3)
            {
                String password = (String)args[3];
                // パスワードのセット
                doc.setPassword(password);
            }

            // PDFファイルをロードします。
            doc.load(inputFile);
            // 暗号化するコンテンツを選択
            System.out.println("暗号化するコンテンツを選択します。");
            System.out.println("暗号化する文書コンポーネントを以下から数値で指定してください。");
            System.out.println("これ以外の数値が指定された場合は 0 とみなされます。");
            System.out.println("0 : 文書の全てのコンテンツを暗号化");
            System.out.println("1 : 文書のメタデータを除く全てのコンテンツを暗号化");
            System.out.println("2 : 添付ファイルのみを暗号化");
            int encComponent = Integer.parseInt(br.readLine());

            System.out.println("256 bit AES で暗号化します。");
            System.out.println("また、全ての権限情報は破棄されます。");
            // 暗号化するコンポーネントを変更。
            encrypt256AESsetEncComponent(doc, outUserPass, encComponent);
            

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }


    public static void encrypt256AESsetEncComponent(PtlPDFDocument doc, String outUserPass,
                                                    int encComponent)
        throws PtlException, IOException, Exception, Error
    {
        // 256 bit AES
        try (PtlEncryptStandard256AES enc256 = new PtlEncryptStandard256AES())
        {
            switch(encComponent)
            {
            case 0:
            default:
                System.out.println("全てのコンテンツを暗号化します。");
                enc256.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL);
                break;
            case 1:
                System.out.println("メタデータを除く全てのコンテンツを暗号化します。");
                enc256.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_EXCEPT_METADATA);
                break;
            case 2:
                System.out.println("添付ファイルのみ暗号化します。");
                System.out.print("in-pdf-file に添付ファイルが設定されていない場合、");
                System.out.println("「0 : 全てのコンテンツを暗号化」と同じ処理がなされます。");
                enc256.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ONLY_ATTACHED_FILE);
                break;
            }

            enc256.setUserPassword(outUserPass);
            // 暗号化情報の設定
            doc.setEncrypt(enc256);
        }
    }
    
}