/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：編集不可セキュリティの設定
  　　　編集の可否について設定します。
  　　　暗号化済みのファイルの場合はその他のセキュリティ情報は保持されます。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;

public class EncryptSetModify {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            System.out.print("usage: java EncryptSetModify in-pdf-file out-pdf-file ");
            System.out.println("out-pdf-ownerpass [in-pdf-password]");
            return;
        }

        // コマンドライン引数の取得
        String outOwnerPass = args[2];

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            if (args.length > 3) {
                String password = (String)args[3];
                // パスワードのセット
                doc.setPassword(password);
            }

            // PDFファイルをロードします。
            doc.load(inputFile);

            // 暗号化の有無判定
            if (doc.isEncrypted()) {
                // 暗号化情報の取得
                setPermModify(doc, outOwnerPass, br);
            }
            else {
                System.out.println("暗号化されていないファイルです。\n");
                System.out.println("変更権限を設定して暗号化します。");
                System.out.println("暗号化種類の設定をします。0から2の数値を指定してください。");
                System.out.println("それ以外の数値が設定された場合は 2 が選択されたものとして扱います。");
                System.out.println("暗号化種類");
                System.out.println("0 : 128 bit RC4  1 : 128 bit AES  2 : 256 bit AES");
                int kind = Integer.parseInt(br.readLine());
                // セキュリティ設定
                switch (kind) {
                    case 0:
                        // 128 bit RC4
                        encrypt128RC4setModify(doc, outOwnerPass, br);
                        break;
                    case 1:
                        // 128 bit AES
                        encrypt128AESsetModify(doc, outOwnerPass, br);
                        break;
                    case 2:
                    default:
                        // 256 bit AES
                        encrypt256AESsetModify(doc, outOwnerPass, br);
                        break;
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }


    public static void encrypt40RC4setModify(PtlPDFDocument doc, String outOwnerPass, BufferedReader br)
        throws PtlException, IOException, Exception, Error {
        // 40 bit RC4
        System.out.println("40 bit RC4暗号化が選択されました。");
        try (PtlEncryptStandard40RC4 enc40 = new PtlEncryptStandard40RC4()) {
            setModifyWithDefaultPermission(doc, outOwnerPass, br, enc40);
        }
    }

    public static void encrypt128RC4setModify(PtlPDFDocument doc, String outOwnerPass, BufferedReader br)
        throws PtlException, IOException, Exception, Error {
        // 128 bit RC4
        System.out.println("128 bit RC4暗号化が選択されました。");
        try (PtlEncryptStandard128RC4 enc128r = new PtlEncryptStandard128RC4()) {
            setModifyWithDefaultPermission(doc, outOwnerPass, br, enc128r);
        }
    }

    public static void encrypt128AESsetModify(PtlPDFDocument doc, String outOwnerPass, BufferedReader br)
        throws PtlException, IOException, Exception, Error {
        // 128 bit AES
        System.out.println("128 bit AES暗号化が選択されました。");
        try (PtlEncryptStandard128AES enc128a = new PtlEncryptStandard128AES()) {
            setModifyWithDefaultPermission(doc, outOwnerPass, br, enc128a);
        }
    }

    public static void encrypt256AESsetModify(PtlPDFDocument doc, String outOwnerPass, BufferedReader br)
        throws PtlException, IOException, Exception, Error {
        // 256 bit AES
        System.out.println("256 bit AES暗号化が選択されました。");
        try (PtlEncryptStandard256AES enc256 = new PtlEncryptStandard256AES()) {
            setModifyWithDefaultPermission(doc, outOwnerPass, br, enc256);
        }
    }
    
    public static void setModifyWithDefaultPermission(PtlPDFDocument doc, String outOwnerPass,
                                                      BufferedReader br, PtlEncryptStandard enc)
        throws PtlException, IOException, Exception, Error {

        // 暗号化するコンテンツ
        // 文書の全てのコンテンツを暗号化します。
        enc.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL);
        // オーナーパスワード値の設定
        enc.setOwnerPassword(outOwnerPass);

        // PtlEncryptPermissionType2で権限設定を行う
        try (PtlEncryptPermissionType2 perms2 = new PtlEncryptPermissionType2()) {
            // 変更権限の設定
            String permModify2 = setModifyForPermType2(br);
            perms2.setModify(PtlEncryptPermissionType2.PERMISSION_MODIFY.valueOf(permModify2));

            // 印刷権限の設定 PERM_PRINT_HIGH : 高解像度での印刷を許可
            perms2.setPrint(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_HIGH);
            // テキスト、画像、その他の内容のコピーを有効にするかどうかの設定 true: 有効にする
            perms2.setCopy(true);
            // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの設定
            // true: 有効にする
            perms2.setAccessibility(true);

            // ユーザアクセス許可フラグの設定
            enc.setPermission(perms2);
        }

        // 暗号化情報の設定
        doc.setEncrypt(enc);
    }

    public static void setPermModify(PtlPDFDocument doc,String outOwnerPass, BufferedReader br)
        throws PtlException, IOException, Exception, Error {
        try (PtlEncrypt encrypt = doc.getEncrypt()) {
            
            // 標準セキュリティハンドラのメソッド取得
            if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_STANDARD) {
                // getFilterType()に判定済みなので PtlEncryptStandard にダウンキャストする。
                try (PtlEncryptStandard encryptStandard = (PtlEncryptStandard)encrypt;
                        PtlEncryptPermission perms = encryptStandard.getPermission()) { // ユーザアクセス許可フラグの取得
                    // タイプ取得
                    switch (perms.getType()) {
                    case PERM_TYPE1:
                        // Type 1は40bit RC4なので、128bit RC4で新規に権限を設定する。
                        System.out.println("入力ファイルは40 bit RC4暗号で暗号化されています。");
                        System.out.println("新規に暗号化します。");
                        encrypt128RC4setModify(doc, outOwnerPass, br);
                        break;
                        
                    case PERM_TYPE2:
                        // getType()で判定されたのでpermsをダウンキャストできる。
                        try(PtlEncryptPermissionType2 perms2 = (PtlEncryptPermissionType2)perms) {
                        // 編集権限設定
                            String permModify2 = setModifyForPermType2(br);
                            perms2.setModify(PtlEncryptPermissionType2.PERMISSION_MODIFY.valueOf(permModify2));
                            encryptStandard.setPermission(perms2);
                        }
                        encryptStandard.setOwnerPassword(outOwnerPass);
                        // 暗号化情報の設定
                        doc.setEncrypt(encryptStandard);
                        break;
                    }
                }
            }
            else {
                System.out.println("標準セキュリティハンドラ以外のセキュリティハンドラが使われたPDFです。");
                System.out.println("このPDFの変更権限を変更することはできません。");
                return;
            }
        }
    }

    public static String setModifyForPermType2(BufferedReader br)
        throws PtlException, IOException, Exception, Error {
        System.out.println("変更権限の設定をします。以下から数値で指定してください。");
        System.out.println("これ以外の数値が指定された場合は 0 とみなされます。");
        System.out.println("0 : 許可しない");
        System.out.println("1 : 注釈の作成、フォームフィールドの入力と既存の署名フィールドに署名を許可");
        System.out.println("2 : ページの挿入、削除、回転を許可");
        System.out.println("3 : フォームフィールドの入力と既存の署名フィールドに署名を許可");
        System.out.println("4 : ページ抽出を除く全ての動作を許可");

        int choiceOfPerm = Integer.parseInt(br.readLine());
        String permModify2 = "PERM_MODIFY_NOT_ALLOWED";
        switch (choiceOfPerm) {
        case 0:
            // PERM_MODIFY_NOT_ALLOWEDを用いるため、変更無し
            break;
        case 1:
            permModify2 = "PERM_MODIFY_ANNOTANDFORM";
            break;
        case 2:
            permModify2 = "PERM_MODIFY_ASSEMBLEDOC";
            break;
        case 3:
            permModify2 = "PERM_MODIFY_FILLFORM";
            break;
        case 4:
            permModify2 = "PERM_MODIFY_MODYFYDOC";
            break;
        default:
            // PERM_MODIFY_NOT_ALLOWEDを用いるため、変更無し
            break;
        }
        
        return permModify2;
    }
}

