/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：セキュリティ情報の設定・ファイルを開くためのユーザーパスワードのみ設定する。
  　　　すべてのユーザアクセス許可フラグの情報は破棄される。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class EncryptWithUserPass {

    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.print("usage: java EncryptWithUserPass in-pdf-file out-pdf-file ");
        System.out.println("暗号化種類 out-pdf-userpass [in-pdf-password]\n");
        System.out.println("暗号化種類");
        System.out.println("0 : 128 bit RC4  1 : 128 bit AES  2 : 256 bit AES");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            printUsage(); // usageメッセージの表示
            return;
        }

        int kind;
        try {
            kind =  Integer.parseInt(args[2]);
            switch (kind) {
            case 0:
            case 1:
            case 2:
                break;
            default:
                System.out.println("暗号化種類には、0から2の数値を指定してください。");
                printUsage(); // usageメッセージの表示
                return;
            }
        }
        catch(NumberFormatException numfe) {
            System.out.println("暗号化種類には、0から2の数値を指定してください。");
            printUsage(); // usageメッセージの表示
            return;
        }
        // コマンドライン引数の取得
        String outUserPass = args[3];
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            if (args.length > 4)
            {
                String password = (String)args[4];
                // パスワードのセット
                doc.setPassword(password);
            }

            // PDFファイルをロードします。
            doc.load(inputFile);
            
            // セキュリティ設定
            switch (kind) {
            case 0:
                // 128 bit RC4
                encrypt128RC4withUserPass(doc, outUserPass);
                break;
            case 1:
                // 128 bit AES
                encrypt128AESwithUserPass(doc, outUserPass);
                break;
            case 2:
                // 256 bit AES
                encrypt256AESwithUserPass(doc, outUserPass);
                break;
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void encrypt128RC4withUserPass(PtlPDFDocument doc, String outUserPass)
        throws PtlException, Exception, Error {
        // 128 bit RC4
        System.out.println("128 bit RC4暗号化の設定をします。");
        try (PtlEncryptStandard128RC4 enc128r = new PtlEncryptStandard128RC4()) {
            setEncryptWithUserPass(doc, outUserPass, enc128r);
        }
    }

    public static void encrypt128AESwithUserPass(PtlPDFDocument doc, String outUserPass)
        throws PtlException, Exception, Error {
        // 128 bit AES
        System.out.println("128 bit AES暗号化の設定をします。");
        try (PtlEncryptStandard128AES enc128a = new PtlEncryptStandard128AES()) {
            setEncryptWithUserPass(doc, outUserPass, enc128a);
        }
    }

    public static void encrypt256AESwithUserPass(PtlPDFDocument doc, String outUserPass)
        throws PtlException, Exception, Error {
        // 256 bit AES
        System.out.println("256 bit AES暗号化の設定をします。");
        try (PtlEncryptStandard256AES enc256 = new PtlEncryptStandard256AES()) {
            setEncryptWithUserPass(doc, outUserPass, enc256);
        }
    }

    public static void setEncryptWithUserPass(PtlPDFDocument doc, String outUserPass, PtlEncryptStandard enc)
        throws PtlException, Exception, Error {
        // 暗号化するコンテンツを設定
        // ENCRYPT_ALL : 全て暗号化
        enc.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL);
        // ユーザパスワード値の設定
        enc.setUserPassword(outUserPass);

        doc.setEncrypt(enc);
    }
}

