/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：セキュリティ情報の取得

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class GetEncryptInfo {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("usage: java GetEncryptInfo in-pdf-file [in-pdf-password]");
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            if (args.length > 1)
            {
                // パスワードのセット
                doc.setPassword(args[1]);
            }

            // PDFファイルをロードします。
            doc.load(inputFile);

            // 暗号化の取得
            if (doc.isEncrypted()) {
                // 暗号化情報の取得
                showEncrypt(doc);
            }
            else {
                System.out.println("暗号化されていないファイルです");
            }
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void showEncrypt(PtlPDFDocument doc) throws PtlException, Exception, Error
    {
        try (PtlEncrypt encrypt = doc.getEncrypt()) {
            // キー長の取得
            System.out.println("暗号化のキー長 : " + encrypt.getKeyLength());

            // 暗号化する文書コンポーネント取得
            switch (encrypt.getEncryptComponent()) {
                case ENCRYPT_ALL:
                    System.out.println("文書の全てのコンテンツを暗号化");
                    break;
                case ENCRYPT_EXCEPT_METADATA:
                    System.out.println("文書のメタデータを除く全てのコンテンツを暗号化");
                    break;
                case ENCRYPT_ONLY_ATTACHED_FILE:
                    System.out.println("添付ファイルのみを暗号化(AES暗号化のみで有効)");
                    break;
            }

            // 暗号化のフィルター取得
            // FILTER_STANDARD = 1 /* 標準セキュリティハンドラ */
            // Ver5.0 はStandardしかない
            if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_STANDARD) {
                // getFilterType()に判定済みなので PtlEncryptStandard にダウンキャストする。
                PtlEncryptStandard encryptStandard = (PtlEncryptStandard)encrypt;

                // 標準セキュリティハンドラのメソッド取得
                switch (encryptStandard.getMethod())
                {
                case METHOD_RC4:
                    System.out.println("標準セキュリティハンドラのメソッド : RC4");
                    break;
                case METHOD_AES:
                    System.out.println("標準セキュリティハンドラのメソッド : AES");
                    break;
                }

                // ユーザアクセス許可フラグの取得
                try (PtlEncryptPermission perms = encryptStandard.getPermission())
                {
                    // タイプ取得
                    switch (perms.getType())
                    {
                    case PERM_TYPE1:
                    {
                        System.out.println("標準セキュリティハンドラのリビジョン : 2");
                        // getType()で判定されたのでpermsをダウンキャストできる。
                        PtlEncryptPermissionType1 perms1 = (PtlEncryptPermissionType1)perms;
                        // 印刷権限取得
                        if (perms1.getPrint())
                        {
                            System.out.println("印刷を許可する");
                        }
                        else
                        {
                            System.out.println("印刷を許可しない");
                        }
                        // 変更権限取得
                        switch (perms1.getModify())
                        {
                        case PERM_MODIFY_NOT_ALLOWED:
                            System.out.println("変更権限 : 許可しない");
                            break;
                        case PERM_MODIFY_ANNOTANDFORM:
                            System.out.println("変更権限 : 注釈の作成、フォームフィールドの入力と既存の署名フィールドに署名");
                            break;
                        case PERM_MODIFY_ASSEMBLEANDFORM:
                            System.out.println("変更権限 : ページレイアウト、フォームフィールドの入力と既存の署名フィールドに署名");
                            break;
                        case PERM_MODIFY_MODYFYDOC:
                            System.out.println("変更権限 : ページ抽出を除く全ての動作");
                            break;
                        }
                        // テキスト、画像、その他の内容のコピーおよびアクセシビリティを有効にするかどうかの取得
                        if (perms1.getCopy())
                        {
                            System.out.println("テキスト、画像、その他の内容のコピーおよびアクセシビリティを有効にする");
                        }
                        else
                        {
                            System.out.println("テキスト、画像、その他の内容のコピーおよびアクセシビリティを有効にしない");
                        }
                    }
                    break;

                    case PERM_TYPE2:
                    {
                        System.out.println("標準セキュリティハンドラのリビジョン : 3または4");
                        // getType()で判定されたのでpermsをダウンキャストできる。
                        PtlEncryptPermissionType2 perms2 = (PtlEncryptPermissionType2)perms;
                        // 印刷権限取得
                        switch (perms2.getPrint())
                        {
                        case PERM_PRINT_NOT_ALLOWED:
                            System.out.println("印刷権限 : 許可しない");
                            break;
                        case PERM_PRINT_LOW:
                            System.out.println("印刷権限 : 低解像度");
                            break;
                        case PERM_PRINT_HIGH:
                            System.out.println("印刷権限 : 高解像度");
                            break;
                        }
                        // 変更権限取得
                        switch (perms2.getModify())
                        {
                        case PERM_MODIFY_NOT_ALLOWED:
                            System.out.println("変更権限 : 許可しない");
                            break;
                        case PERM_MODIFY_ASSEMBLEDOC:
                            System.out.println("変更権限 : ページの挿入、削除、回転");
                            break;
                        case PERM_MODIFY_FILLFORM:
                            System.out.println("変更権限 : フォームフィールドの入力と既存の署名フィールドに署名");
                            break;
                        case PERM_MODIFY_ANNOTANDFORM:
                            System.out.println("変更権限 : 注釈の作成、フォームフィールドの入力と既存の署名フィールドに署名");
                            break;
                        case PERM_MODIFY_MODYFYDOC:
                            System.out.println("変更権限 : ページ抽出を除く全ての動作");
                            break;
                        }

                        // テキスト、画像、その他の内容のコピーを有効にするかどうかの取得
                        if (perms2.getCopy())
                        {
                            System.out.println("テキスト、画像、その他の内容のコピーを有効にする");
                        }
                        else
                        {
                            System.out.println("テキスト、画像、その他の内容のコピーを有効にしない");
                        }
                        // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの取得
                        if (perms2.getAccessibility())
                        {
                            System.out.println("スクリーンリーダーデバイスのテキストアクセスを有効にする");
                        }
                        else
                        {
                            System.out.println("スクリーンリーダーデバイスのテキストアクセスを有効にしない");
                        }
                    }
                    break;
                    }
                }
            }
        }
    }
}

