/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：しおり情報の取得
  　　　
  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class GetOutlineInfo {
    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.println("usage: java GetOutline in-pdf-file 取得方法\n");
        System.out.println("取得方法\n0 : ツリー   1 : 詳細");
    }


    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 2) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の処理
        boolean showDetail;
        try {
            showDetail = readBoolArgs(args[1], "取得方法は0か1で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }

        // しおり情報の取得
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロード
            doc.load(inputFile);

            if (!doc.hasOutlines()) {
                System.out.println("このPDFにしおり情報はありません。");
                return;
            }

            try (PtlOutline rootOutline = doc.getRootOutline()) {
                if (rootOutline.hasChild()) {
                    try (PtlOutline outline = rootOutline.getFirstChild()) {
                        readOutline(outline, 0, showDetail);
                    }
                }
            }
        }
         catch(PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch(Exception ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        catch (Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    static void readOutline(PtlOutline item, int indent, boolean showDetail)
        throws PtlException, Exception, Error {
        if (showDetail) {
            System.out.println("Title = " + item.getTitle());
            System.out.println("Indent = " + indent);
            if (item.isOpen()) {
                System.out.println("PDF表示時の子アウトラインをオープンする");
            }
            else {
                System.out.println("PDF表示時の子アウトラインをオープンしない");
            }
            int flags = item.getFlags();
            if ((flags & PtlOutline.FLAG_BOLD) == PtlOutline.FLAG_BOLD)
                System.out.println("FLAG_BOLD on" );
            else
                System.out.println("FLAG_BOLD off" );
            if ((flags & PtlOutline.FLAG_ITALIC) == PtlOutline.FLAG_ITALIC)
                System.out.println("FLAG_ITALIC on" );
            else
                System.out.println("FLAG_ITALIC off" );
            try (PtlColorDeviceRGB color = item.getColor()) {
                int rr = (int)(color.getR() * 255);
                int gg = (int)(color.getG() * 255);
                int bb = (int)(color.getB() * 255);
                System.out.printf("Color = RGB(%d,%d,%d)\n", rr, gg, bb);
            }
            try (PtlAction act = item.getAction()) {
                showAction(act);
            }
        } else {
            String indentString = "";
            for(int i=0; i<indent; ++i) {
                indentString += "  ";
            }
            if (showDetail == false) {
                System.out.print(indentString);
            }
            System.out.println(item.getTitle());
        }

        if (item.hasChild()) {
            try (PtlOutline firstItem = item.getFirstChild()) {
                int indentChild = indent + 1;
                readOutline(firstItem, indentChild, showDetail);
            }
        }
        if (item.hasNextSibling()) {
            try (PtlOutline nextItem = item.getNextSibling()) {
                readOutline(nextItem, indent, showDetail);
            }
        }
    }

    public static void showAction(PtlAction action)
        throws PtlException, Exception, Error {
        switch (action.getType()) {
            case TYPE_NONE: {
                System.out.println("Action = " + PtlAction.ACTION_TYPE.TYPE_NONE +
                                   " : アクションなし");
                break;
            }
            case TYPE_GOTO: {
                System.out.println("Action = " + PtlAction.ACTION_TYPE.TYPE_GOTO +
                                   " : GoToアクション");
                PtlActionGoTo act = (PtlActionGoTo)action;
                // 宛先の取得
                try (PtlDest dest = act.getDest())
                {
                    showDest(dest);
                }
                break;
            }
            case TYPE_GOTO_R: {
                System.out.println("Action = " + PtlAction.ACTION_TYPE.TYPE_GOTO_R +
                                   " : GoToRアクション");
                PtlActionGoToR act = (PtlActionGoToR)action;
                // ファイル間移動用PDFファイル名を取得 getFileName()
                System.out.println("  FileName = " + act.getFileName());
                // 新ウィンドウフラグを取得
                if (act.getNewWindowFlag()) {
                    System.out.println("  NewWindowFlag = true: 新ウィンドウでオープンする");
                }
                else {
                    System.out.println("  NewWindowFlag = false: しない");
                }
                // 宛先の取得
                try (PtlDest dest = act.getDest()) {
                    showDest(dest);
                }
                break;
            }
            case TYPE_LAUNCH: {
                System.out.println("Action = " + PtlAction.ACTION_TYPE.TYPE_LAUNCH +
                                   " : Launchアクション");
                PtlActionLaunch act = (PtlActionLaunch)action;
                // 起動ファイル名を取得
                System.out.println("  FileName = " + act.getFileName());
                // 新ウィンドウフラグを取得
                if (act.getNewWindowFlag()) {
                    System.out.println("  NewWindowFlag = true: 新ウィンドウでオープンする");
                }
                else {
                    System.out.println("  NewWindowFlag = false: しない");
                }
                break;
            }
            case TYPE_URI: {
                System.out.println("Action = " + PtlAction.ACTION_TYPE.TYPE_URI +
                                   " : URIアクション");
                PtlActionURI act = (PtlActionURI)action;
                // URIを取得
                System.out.println("  URI = " + act.getURI());
                break;
            }
            case TYPE_UNKNOWN: {
                System.out.println("Action = " + PtlAction.ACTION_TYPE.TYPE_UNKNOWN +
                                   " : 未対応アクション");
                break;
            }
        }
    }

    
    public static void showDest(PtlDest dest) throws PtlException, Exception, Error {
        switch (dest.getType()) {
        case TYPE_NONE:
            System.out.println("Dest = " + PtlDest.DEST_TYPE.TYPE_NONE + " : 宛先なし");
            break;
        case TYPE_XYZ:
            System.out.println("Dest = " + PtlDest.DEST_TYPE.TYPE_XYZ + " : XYZ型");
            break;
        case TYPE_FIT:
            System.out.println("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT + " : Fit型(全体表示)");
            break;
        case TYPE_FIT_H:
            System.out.println("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_H +
                               " : FitH型(幅に合わせる)");
            break;
        case TYPE_FIT_V:
            System.out.println("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_V +
                               " : FitV型(高さに合わせる)");
            break;
        case TYPE_FIT_R:
            System.out.println("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_R + " : FitR型");
            break;
        case TYPE_FIT_B:
            System.out.println("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_B + " : FitB型");
            break;
        case TYPE_FIT_BH:
            System.out.println("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_BH +
                               " : FitBH型(描画領域の幅に合わせる)");
            break;
        case TYPE_FIT_BV:
            System.out.println("Dest = " + PtlDest.DEST_TYPE.TYPE_FIT_BV + " : FitBV型");
            break;
        }
        // 宛先ページの取得
        System.out.println("宛先ページ(PageNumber) : " + (dest.getPageNumber()) + " ページ目");
    }

    
    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
        throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }

}
