/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：PDF 文書の指定したページに画像透かしを挿入します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class ImageWaterMarkAppend {
    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.print("usage: java ImageWaterMarkAppend in-pdf-file out-pdf-file");
        System.out.println(" watermark-image");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の読み取り
        String pathImage = args[2];
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            // 透かしの追加
            appendImageWaterMark(doc, pathImage);
            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void appendImageWaterMark(PtlPDFDocument doc, String pathImage)
        throws PtlException, Exception, Error {
        try (PtlParamWaterMarkImage waterMarkImage = new PtlParamWaterMarkImage())
        {
            // 透かしに指定する画像の設定
            try (PtlParamInput inputImage = new PtlParamInput(pathImage)) {  // 画像ファイル
                // 画像描画パラメータに画像ファイルを設定
                waterMarkImage.setImageStream(inputImage);
            }
            // 透かしの倍率の設定
            waterMarkImage.setScale(0.8f);


            // 以下、画像透かし以外と共通するデフォルトの設定
            // 透かしの名前の設定
            waterMarkImage.setName("waterMarkImage");
            // 透かしを配置するときの余白の設定
            waterMarkImage.setMargin(10.0f, 10.0f, 10.0f, 10.0f);
            // 透かしの配置の設定
            waterMarkImage.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
            // 透かしのZオーダーの設定 ZORDER_FRONT = 1 
            waterMarkImage.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
            // 透かしの不透明度の設定
            waterMarkImage.setOpacity(0.5f);
            // 透かしをタイリングして配置するかどうかの設定
            // false : タイリングして配置しない
            waterMarkImage.setTiling(false);
            // デフォルトの設定終わり

            // 透かしを文書に設定する
            doc.appendWaterMark(waterMarkImage);
        }
    }

}
