/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：PDF 文書の指定したページにPDF透かしを挿入します。
  　　　指定したPDFの指定したページを読み取ってそれを透かしとします。
  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class PdfWaterMarkAppend {
    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.print("usage: java PdfWaterMarkAppend in-pdf-file out-pdf-file");
        System.out.println(" watermark-pdf page-num");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の読み取り
        String pathPdf = args[2];
        int pageNum = Integer.parseInt(args[3]);
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            // 透かしの追加
            appendPdfWaterMark(doc, pathPdf, pageNum);
            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void appendPdfWaterMark(PtlPDFDocument doc, String pathPdf,
                                          int pageNum)
        throws PtlException, Exception, Error {
        try (PtlParamWaterMarkPDF waterMarkPdf = new PtlParamWaterMarkPDF()) {
            try (PtlPDFDocument doc_watermark = new PtlPDFDocument();
                 PtlParamInput inputPdf = new PtlParamInput(pathPdf)) { // PDFファイル
                // PDFファイルをロード
                doc_watermark.load(inputPdf);

                // 透かしに使用するページの取得
                try (PtlPages pages = doc_watermark.getPages()) { //ページコンテナの取得
                    //ページコンテナが空かどうか
                    if (pages.isEmpty()) {
                        System.out.println("ページコンテナが空\n");
                        return;
                    }
                    try (PtlPage page = pages.get(pageNum - 1)) {// 指定したページ
                        // 透かしに使用するPDF文書ページを設定
                        waterMarkPdf.setPage(page);
                    }
                }
            }
            // 透かしの倍率の設定
            waterMarkPdf.setScale(0.0f);


            // 以下、PDF透かし以外と共通するデフォルトの設定
            // 透かしの名前の設定
            waterMarkPdf.setName("waterMarkPdf");
            // 透かしを配置するときの余白の設定
            waterMarkPdf.setMargin(10.0f, 10.0f, 10.0f, 10.0f);
            // 透かしの配置の設定
            waterMarkPdf.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
            // 透かしのZオーダーの設定 ZORDER_FRONT = 1 
            waterMarkPdf.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
            // 透かしの不透明度の設定
            waterMarkPdf.setOpacity(0.5f);
            // 透かしをタイリングして配置するかどうかの設定
            // false : タイリングして配置しない
            waterMarkPdf.setTiling(false);
            // デフォルトの設定終わり

            // 透かしを文書に設定する
            doc.appendWaterMark(waterMarkPdf);
        }
    }

}
