 /*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：閲覧場所の制限設定(URLで指定)

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class SetRestrictionByURI {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            System.out.print("usage: java SetRestrictionByURI in-pdf-file out-pdf-file ");
            System.out.println("out-pdf-ownerpass [in-pdf-password]");
            System.out.println("in-pdf-fileが暗号化されていない場合、40bitRC4で暗号化されていた場合は"
                             + "128 bit AES で暗号化します。");
            return;
        }

        // コマンドライン引数の取得
        String outOwnerPass = args[2];
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            if (args.length > 3) {
                String password = (String)args[3];
                // パスワードのセット
                doc.setPassword(password);
            }

            // PDFファイルをロードします。
            doc.load(inputFile);

            // 閲覧場所の制限設定
            setRestrictionByURI(doc);

            if (doc.isEncrypted()) {
                if(doc.getEncrypt().getKeyLength() >= 128){ //キー長が128以上の場合は設定を引き継げる
                    // 暗号化情報の取得
                    System.out.println("暗号化されたファイルです。設定を引き継ぎます。");
                    setOwnerPass(doc, outOwnerPass, doc.getEncrypt());
                }
                else { //V7.0以降、キー長が40の場合はこのまま暗号化できない。
                    System.out.println("入力ファイルが128bitより短いキー長で暗号化されています。");
                    System.out.println("新規に暗号化します。");
                    encrypt128AESwithOwnerpass(doc, outOwnerPass);
                }
            }
            else {
                System.out.println("暗号化されていないファイルです。");
                encrypt128AESwithOwnerpass(doc, outOwnerPass);
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void setRestrictionByURI(PtlPDFDocument doc)
        throws PtlException, Exception, Error {
        try (PtlParamRestriction restriction = new PtlParamRestriction()) {
            // 閲覧可能とするURIの追加
            restriction.appendValidURL("C:\\temp");

            // 透かしの設定
            setWaterMarkToRestriction(restriction);

            // 閲覧制限の設定
            doc.setRestriction(restriction);
        }
    }

    public static void setWaterMarkToRestriction(PtlParamRestriction restriction)
        throws PtlException, Exception, Error {
        try (PtlParamWaterMarkText watermarktext = new PtlParamWaterMarkText()) {
            // 透かしの配置の設定 ALIGN_CENTER = 5 /* 中央 */
            watermarktext.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);

            // 透かしに指定する文字列の設定
            watermarktext.setString("Restriction Test");

            // 透かしに指定するフォントの設定
            try (PtlParamFont font = new PtlParamFont("Helvetica", 180.0f,
                                                      PtlParamFont.WEIGHT.WEIGHT_HEAVY,
                                                      true, false)) {
                watermarktext.setFont(font);
            }

            // 透かしの文字に指定する色の設定
            try (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f)) {
                watermarktext.setTextColor(colorText);
            }

            // 透かしの文字の縁取りに指定する色の設定
            try (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(0.0f, 1.0f, 0.0f)) {
                watermarktext.setOutlineColor(colorOutline);
            }

            // 透かしのテキストを対角線上に配置する設定
            watermarktext.setWriteDiagonal(true);

            // 閲覧不可時に表示するウォーターマーク設定
            restriction.setWatermark(watermarktext);
        }
    }


    public static void encrypt128AESwithOwnerpass(PtlPDFDocument doc, String outOwnerPass)
        throws PtlException, Exception, Error {
        // 128 bit AES
        System.out.println("128 bit AES形式で暗号化します。");
        try (PtlEncryptStandard128AES enc128a = new PtlEncryptStandard128AES()) {
            setOwnerPass(doc, outOwnerPass, enc128a);
        }
    }

    public static void setOwnerPass(PtlPDFDocument doc, String outOwnerPass, PtlEncrypt encrypt)
        throws PtlException, Exception, Error {
            
        // 標準セキュリティハンドラのメソッド取得
        if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_STANDARD) {
            // getFilterType()に判定済みなので PtlEncryptStandard にダウンキャストする。
            try(PtlEncryptStandard encryptStandard = (PtlEncryptStandard)encrypt) {
                encryptStandard.setOwnerPassword(outOwnerPass);
                // 暗号化情報の設定
                doc.setEncrypt(encryptStandard);
            }
        }
        else {
            System.out.println("標準セキュリティハンドラ以外のセキュリティハンドラが使われたPDFです。");
            System.out.println("このPDFの暗号化権限を引き継ぐことはできません。");
        }
    }

    
}
