/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：PDF 文書の指定したページにテキストを透かしとして挿入します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class TextWaterMarkAppend {
    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.print("usage: java TextWaterMarkAppend in-pdf-file out-pdf-file");
        System.out.println(" watermark-text watermark-page-range");
        System.out.println("watermark-page-range:");
        System.out.println("透かしを配置するページの範囲を指定する文字列。0が先頭。");
        System.out.println("例:『0,2-4』 と指定すると、1ページと3から5ページに透かしが入ります。");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の読み取り
        String contentOfWaterMark = args[2];
        String pageRange = args[3];
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            // 透かしの追加
            appendTextWaterMark(doc, contentOfWaterMark, pageRange);
            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void appendTextWaterMark(PtlPDFDocument doc, String contentOfWaterMark,
                                           String pageRange)
        throws PtlException, Exception, Error {
        try (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText()) {
            // 透かしに指定する文字列の設定
            waterMarkText.setString(contentOfWaterMark);
            // 透かしを入れるページの範囲の設定 PAGE_RANGE_CUSTOM : ページ範囲指定
            waterMarkText.setCustomPageRange(pageRange);
            waterMarkText.setPageRange(PtlParamWaterMark.PAGE_RANGE.PAGE_RANGE_CUSTOM);


            // 以下、その他の項目について透かしであることが分かりやすいようデフォルトの設定をする
            // 透かしの名前の設定
            waterMarkText.setName("waterMarkText");
            // 透かしの配置の設定
            waterMarkText.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
            // 透かしのZオーダーの設定 ZORDER_FRONT = 1 
            waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
            // 透かしの不透明度の設定
            waterMarkText.setOpacity(1.0f);
            // 透かしをタイリングして配置するかどうかの設定
            // false : タイリングして配置しない
            waterMarkText.setTiling(false);
            // 透かしに指定するフォントの設定
            try (PtlParamFont font = new PtlParamFont("Times New Roman", 72.0f,
                                                      PtlParamFont.WEIGHT.WEIGHT_MEDIUM, false, true)) {
                waterMarkText.setFont(font);
            }
            // 透かしの文字に指定する色の設定
            try (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f)) {
                waterMarkText.setTextColor(colorText);
            }
            // 透かしの文字の縁取りに指定する色の設定
            try (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(0.0f, 0.0f, 0.0f)) {
                waterMarkText.setOutlineColor(colorOutline);
            }
            // 透かしのテキストを対角線上に配置する設定
            // false : 対角線上に配置しない
            waterMarkText.setWriteDiagonal(false);
            // 透かしのテキストを任意の角度で配置する設定
            waterMarkText.setTextAngle(15.0f);
            // デフォルトの設定終わり

            // 透かしを文書に設定する
            doc.appendWaterMark(waterMarkText);
        }
    }

}
