/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：透かし文字列のアウトラインの色を指定する。
  　　　今回は色の指定に色空間PtlColorDeviceRGBを使用。
  　　　色空間はPtlColorNone, PtlColorDeviceGray, PtlColorDeviceCMYKに
  　　　置き換えて指定することも可能。その場合は指定する変数の種類に注意。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class TextWaterMarkSetOutlineColor {
    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.print("usage: java TextWaterMarkSetOutlineColor in-pdf-file out-pdf-file");
        System.out.println(" outline-color-R outline-color-G outline-color-B");
        System.out.println("outline-color-R, outline-color-G, outline-color-B :");
        System.out.println("0.0～1.0の数値で指定。それぞれの色について1.0が最大");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 5) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の読み取り
        float colorR = Float.parseFloat(args[2]);
        float colorG = Float.parseFloat(args[3]);
        float colorB = Float.parseFloat(args[4]);

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            // 透かしの追加
            appendTextWaterMarkSetOutlineColor(doc, colorR, colorG, colorB);
            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
        
    public static void appendTextWaterMarkSetOutlineColor(PtlPDFDocument doc, float colorR,
                                                   float colorG, float colorB)
        throws PtlException, Exception, Error {
        try (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText()) {
            // 透かしの文字の縁取りに指定する色の設定
            // PtlColorDeviceRGBは、PtlColorNone, PtlColorDeviceGray, PtlColorDeviceCMYKに
            // 置き換えて指定することも可能。その場合は指定する変数の種類に注意。
            try (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(colorR, colorG, colorB)) {
                waterMarkText.setOutlineColor(colorOutline);
            }


            // 以下、その他の項目について透かしであることが分かりやすいようデフォルトの設定をする
            // 透かしの名前の設定
            waterMarkText.setName("waterMarkText");
            // 透かしの配置の設定
            waterMarkText.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
            // 透かしのZオーダーの設定 ZORDER_FRONT = 1 
            waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
            // 透かしの不透明度の設定
            waterMarkText.setOpacity(1.0f);
            // 透かしをタイリングして配置するかどうかの設定
            // false : タイリングして配置しない
            waterMarkText.setTiling(false);
            // 透かしに指定する文字列の設定
            waterMarkText.setString("WaterMarkText test");
            // 透かしに指定するフォントの設定
            try (PtlParamFont font = new PtlParamFont("Times New Roman", 72.0f,
                                                      PtlParamFont.WEIGHT.WEIGHT_MEDIUM,
                                                      false, true)) {
                waterMarkText.setFont(font);
            }
            // 透かしの文字に指定する色の設定
            try (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f)) {
                waterMarkText.setTextColor(colorText);
            }
            // 透かしのテキストを対角線上に配置する設定
            // false : 対角線上に配置しない
            waterMarkText.setWriteDiagonal(false);
            // 透かしのテキストを任意の角度で配置する設定
            waterMarkText.setTextAngle(15.0f);
            // デフォルトの設定終わり

            // 透かしを文書に設定する
            doc.appendWaterMark(waterMarkText);
        }
    }

}
