/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：余白の指定をすることで透かしの挿入矩形・配置を指定する

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;


public class WaterMarkSetMargin {
    
    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 7) {
            System.out.print("usage: java WaterMarkSetMargin in-pdf-file out-pdf-file watermark-align ");
            System.out.println("marginTop marginLeft marginBottom marginRight");
            return;
        }
        // コマンドライン引数の取得 
        String waterMarkAlign = args[2];
        float marginTop    = Float.parseFloat(args[3]);
        float marginLeft   = Float.parseFloat(args[4]);
        float marginBottom = Float.parseFloat(args[5]);
        float marginRight  = Float.parseFloat(args[6]);
        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            // 透かしの追加
            appendWaterMarkSetMargin(doc, waterMarkAlign,
                                     marginTop, marginLeft, marginBottom, marginRight);

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() + 
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void appendWaterMarkSetMargin(PtlPDFDocument doc, String waterMarkAlign,
                                                float marginTop, float marginLeft,
                                                float marginBottom, float marginRight)
        throws PtlException, Exception, Error {
        //透かしの設定
        try (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText()) {
            // 透かしの配置の設定
            System.out.println("透かしの配置(watermark-align)を設定します。: " + waterMarkAlign);
            try {
                waterMarkText.setAlign(PtlParamWaterMark.ALIGN.valueOf(waterMarkAlign));
            }
            catch (IllegalArgumentException ex){ // PtlParamWaterMark.ALIGN以外の文字列のエラーメッセージ
                System.out.println(ex.getMessage());
                System.out.print("ERROR : ");
                System.out.println("alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                ex.printStackTrace();
            }
            // 透かしを配置するときの余白の設定
            System.out.println("表示矩形の内側の余白を指定します。");
            System.out.println("marginTop (mm) : " + marginTop);
            System.out.println("marginLeft (mm) : " + marginLeft);
            System.out.println("marginBottom (mm) : " + marginBottom);
            System.out.println("marginRight (mm) : " + marginRight);
            waterMarkText.setMargin(marginTop, marginLeft, marginBottom, marginRight);


            // 配置以外の透かしのデフォルトの設定
            // 透かしの名前の設定
            waterMarkText.setName("nameOfWaterMark");
            // 透かしのZオーダーの設定 ZORDER_FRONT = 1 
            waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
            // 透かしを入れるページの範囲の設定 PAGE_RANGE_ALL = 0 /* 全ページ */
            waterMarkText.setPageRange(PtlParamWaterMark.PAGE_RANGE.PAGE_RANGE_ALL);
            // 透かしの不透明度の設定
            waterMarkText.setOpacity(1.0f);
            // 透かしをタイリングして配置するかどうかの設定
            // false : タイリングして配置しない
            waterMarkText.setTiling(false);
            // 透かしに指定する文字列の設定
            waterMarkText.setString("WaterMark Test");
            // 透かしに指定するフォントの設定
            try (PtlParamFont font = new PtlParamFont("Times New Roman", 72.0f,
                                                      PtlParamFont.WEIGHT.WEIGHT_MEDIUM, false, true)) {
                waterMarkText.setFont(font);
            }
            // 透かしの文字に指定する色の設定
            try (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f)) {
                waterMarkText.setTextColor(colorText);
            }
            // 透かしの文字の縁取りに指定する色の設定
            try (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(0.0f, 1.0f, 0.0f)) {
                waterMarkText.setOutlineColor(colorOutline);
            }
            // 透かしのテキストを対角線上に配置する設定
            // false : 対角線上に配置しない
            waterMarkText.setWriteDiagonal(false);
            // 透かしのテキストを任意の角度で配置する設定
            waterMarkText.setTextAngle(15.0f);
            // デフォルトの設定終わり

            doc.appendWaterMark(waterMarkText);
        }
    }

}
