/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：PDF印刷時に透かしを印刷するかどうかを指定します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class WaterMarkSetPrint {
    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.println("usage: java WaterMarkSetPrint in-pdf-file out-pdf-file print-watermark");
        System.out.println("print-watermark : \n 0 : PDF印刷時に透かしを印刷しない。1 : 印刷する。");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の読み取り・判定
        boolean setPrint;
        try {
            setPrint = readBoolArgs(args[2], "print-watermarkは0か1で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            // 透かしの追加
            appendWaterMarkSetPrint(doc, setPrint);
            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void appendWaterMarkSetPrint(PtlPDFDocument doc, boolean setPrint)
        throws PtlException, Exception, Error {
        try (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText()) {
            // PDF印刷時に透かしを印刷する指定の設定。
            waterMarkText.setPrintWaterMark(setPrint);

            // 以下、その他の項目について透かしであることが分かりやすいようデフォルトの設定をする
            // 透かしの配置の設定
            waterMarkText.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
            // 透かしの名前の設定
            waterMarkText.setName("nameOfWaterMark");
            // 透かしのZオーダーの設定 ZORDER_FRONT = 1 
            waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
            // 透かしの不透明度の設定
            waterMarkText.setOpacity(1.0f);
            // 透かしをタイリングして配置するかどうかの設定
            // false : タイリングして配置しない
            waterMarkText.setTiling(false);
            // 透かしに指定する文字列の設定
            waterMarkText.setString("WaterMark Test");
            // 透かしに指定するフォントの設定
            try (PtlParamFont font = new PtlParamFont("Times New Roman", 72.0f,
                                                      PtlParamFont.WEIGHT.WEIGHT_MEDIUM, false, true)) {
                waterMarkText.setFont(font);
            }
            // 透かしの文字に指定する色の設定
            try (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f)) {
                waterMarkText.setTextColor(colorText);
            }
            // 透かしの文字の縁取りに指定する色の設定
            try (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(0.0f, 0.0f, 0.0f)) {
                waterMarkText.setOutlineColor(colorOutline);
            }
            // 透かしのテキストを対角線上に配置する設定
            // false : 対角線上に配置しない
            waterMarkText.setWriteDiagonal(false);
            // 透かしのテキストを任意の角度で配置する設定
            waterMarkText.setTextAngle(15.0f);
            // デフォルトの設定終わり

            // 透かしを文書に設定する
            doc.appendWaterMark(waterMarkText);
        }
    }

    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
        throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }
    
}
