/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：透かしの挿入矩形・配置を指定する

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;


public class WaterMarkSetRect {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            System.out.println("usage: java WaterMarkSetRect in-pdf-file out-pdf-file watermark-align");
            return;
        }

        String waterMarkAlign = args[2];
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            // 透かしの追加
            appendWaterMarkSetRect(doc, br, waterMarkAlign);

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void appendWaterMarkSetRect(PtlPDFDocument doc, BufferedReader br,
                                              String waterMarkAlign)
        throws PtlException, Exception, Error {
        try (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText()) {

            // 透かしを配置する矩形の設定(setRectCoordinate()を使って改めて指定する。)
            System.out.println("透かしを配置する矩形の設定を行います。");
            try (PtlRect rect = new PtlRect()) {
                waterMarkText.setRect(setRectCoordinate(br, rect));
            }

            // 透かしの配置の設定
            System.out.println("透かしの配置(watermark-align)を設定します。: " + waterMarkAlign);
            try {
                waterMarkText.setAlign(PtlParamWaterMark.ALIGN.valueOf(waterMarkAlign));
            }
            catch (IllegalArgumentException ex){//PtlContent.ALIGNが誤っていた場合のエラーメッセージ
                System.out.println(ex.getMessage());
                System.out.println("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                ex.printStackTrace();
            }

            
            // 以下、その他の項目について透かしであることが分かりやすいようデフォルトの設定をする
            // 透かしの名前の設定
            waterMarkText.setName("nameOfWaterMark");
            // 透かしのZオーダーの設定 ZORDER_FRONT = 1 
            waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
            // 透かしを入れるページの範囲の設定 PAGE_RANGE_ALL = 0 /* 全ページ */
            waterMarkText.setPageRange(PtlParamWaterMark.PAGE_RANGE.PAGE_RANGE_ALL);
            // 透かしの不透明度の設定
            waterMarkText.setOpacity(1.0f);
            // 透かしをタイリングして配置するかどうかの設定
            // false : タイリングして配置しない
            waterMarkText.setTiling(false);
            // 透かしに指定する文字列の設定
            waterMarkText.setString("WaterMark Test");
            // 透かしに指定するフォントの設定
            try (PtlParamFont font = new PtlParamFont("Times New Roman", 72.0f,
                                                      PtlParamFont.WEIGHT.WEIGHT_MEDIUM,
                                                      false, true)) {
                waterMarkText.setFont(font);
            }
            // 透かしの文字に指定する色の設定
            try (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f)) {
                waterMarkText.setTextColor(colorText);
            }
            // 透かしの文字の縁取りに指定する色の設定
            try (PtlColorDeviceRGB colorOutline = new PtlColorDeviceRGB(0.0f, 1.0f, 0.0f)) {
                waterMarkText.setOutlineColor(colorOutline);
            }
            // 透かしのテキストを対角線上に配置する設定
            // false : 対角線上に配置しない
            waterMarkText.setWriteDiagonal(false);
            // 透かしのテキストを任意の角度で配置する設定
            waterMarkText.setTextAngle(15.0f);
            // デフォルトの設定終わり

            // 透かしを文書に設定する
            doc.appendWaterMark(waterMarkText);
        }
    }

    /**
     * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
     * 原点はPDFの左下端。
     * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * @param br BufferedReader。数値の読み取りに使う。
     * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
     * @return 指定したleft, bottom, right, topの数値を持つPtlRect
     * @throws IOException
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    public static PtlRect setRectCoordinate(BufferedReader br, PtlRect outputRect)
           throws IOException, PtlException, Exception, Error {
        float top, bottom, left, right;
        boolean isValueOkay = false;
        
        while(!isValueOkay) {
            System.out.println("指定する矩形の各数値を入力してください。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) {//不正矩形は再入力させる
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            if(right < left) {//不正矩形は再入力させる
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setLeft(left);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setTop(top);
        }
        return outputRect;
    }

}
