﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：画像抽出
  　　　抽出するページ及び出力形式をコマンドライン引数で指定できる。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class ExtractImageSetFormat
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: ExtractImageSetFormat in-pdf-file out-image-file" +
                               " page-to-extract 出力フォーマット");
            Console.WriteLine("出力フォーマット\n0 : 自動  1 : BMP  2 : JPEG  3 : PNG");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 4)
            {
                printUsage();
                return;
            }

            // コマンドライン引数の取得
            int pageToExtract = int.Parse(args[2]);
            int kind = int.Parse(args[3]);
            switch (kind)
            {
                case 0:
                case 1:
                case 2:
                case 3:
                    break;
                default:
                    Console.WriteLine("出力フォーマットは0から3の数値で指定してください。");
                    printUsage();
                    return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    using (PtlPages pages = doc.getPages())
                    { // ページコンテナの取得
                      // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空");
                            return;
                        }

                        // それ以外であれば指定ページ番号を検索。
                        int wholePageNum = doc.getPageCount();
                        //pageToSearchのエラー処理
                        if (wholePageNum < pageToExtract)
                        {
                            Console.WriteLine("ERROR: page-to-extractはPDFの総ページ数より" +
                                               "小さい値を指定してください。");
                            Console.WriteLine("総ページ数：" + wholePageNum);
                            printUsage();
                            return;
                        }
                        // ページの取得(index番号は0が先頭のため1を引く)
                        using (PtlPage page = pages.get(pageToExtract - 1))
                        {
                            // 画像抽出
                            extractImage(page, args[1], kind);
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void extractImage(PtlPage page, String imagefilename, int kind)
        {
            int numImage = 0;
            // ページコンテント・画像エレメントの取得
            using (PtlContent content = page.getContent())
            using (PtlEditElements elems = content.getEditElements(PtlContent.GET_ELEMENT.GET_IMAGE))
            {
                int numElems = elems.getCount();
                if (numElems == 0) return; // 画像エレメントが無い場合はreturnする

                PtlEditImage.OUTPUT_FORMAT format = PtlEditImage.OUTPUT_FORMAT.FORMAT_AUTO;
                switch (kind)
                {
                    case 1:
                        format = PtlEditImage.OUTPUT_FORMAT.FORMAT_BMP;
                        break;
                    case 2:
                        format = PtlEditImage.OUTPUT_FORMAT.FORMAT_JPEG;
                        break;
                    case 3:
                        format = PtlEditImage.OUTPUT_FORMAT.FORMAT_PNG;
                        break;
                }

                for (int i = 0; i < numElems; ++i)
                {
                    // 画像エレメントの取得
                    using (PtlEditElement elem = elems.get(i))
                    {
                        PtlEditElement.ELEMENT_TYPE type = elem.getType();
                        switch (type)
                        {
                            case PtlEditElement.ELEMENT_TYPE.TYPE_IMAGE:
                                {
                                    ++numImage;
                                    PtlEditImage elemImage = (PtlEditImage)elem;
                                    String pathImage = "";
                                    switch (format)
                                    {
                                        case PtlEditImage.OUTPUT_FORMAT.FORMAT_AUTO:
                                            pathImage = String.Format("{0}{1}", imagefilename, numImage);
                                            break;
                                        case PtlEditImage.OUTPUT_FORMAT.FORMAT_BMP:
                                            pathImage = String.Format("{0}{1}.bmp", imagefilename, numImage);
                                            break;
                                        case PtlEditImage.OUTPUT_FORMAT.FORMAT_JPEG:
                                            pathImage = String.Format("{0}{1}.jpg", imagefilename, numImage);
                                            break;
                                        case PtlEditImage.OUTPUT_FORMAT.FORMAT_PNG:
                                            pathImage = String.Format("{0}{1}.png", imagefilename, numImage);
                                            break;
                                    }
                                    using (PtlParamOutput outputImage = new PtlParamOutput(pathImage))
                                    {
                                        // 画像出力
                                        elemImage.writeFile(outputImage, format);
                                    }
                                }
                                break;
                            default:
                                break;
                        }
                    }
                }
            }
        }
    }
}
