﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：指定されたインデックス番号の画像エレメントを取得し、
  　　　画像エレメントからファイルを書き出します。
  　　　引数で、出力のフォルダパスとファイル名（ボディ部まで）を指定します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class ExtractImageSimple
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: ExtractImageSimple in-pdf-file out-image-body" +
                               " page-to-extract image-index-num");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 4)
            {
                printUsage();
                return;
            }

            // コマンドライン引数の取得
            int pageToExtract = int.Parse(args[2]);
            int imageIndexNum = int.Parse(args[3]);

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    using (PtlPages pages = doc.getPages())
                    { // ページコンテナの取得
                      // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空");
                            return;
                        }

                        // それ以外であれば指定ページ番号を検索。
                        int wholePageNum = doc.getPageCount();
                        //pageToSearchのエラー処理
                        if (wholePageNum < pageToExtract)
                        {
                            Console.WriteLine("ERROR: page-to-extractはPDFの総ページ数より" +
                                               "小さい値を指定してください。");
                            Console.WriteLine("総ページ数：" + wholePageNum);
                            printUsage();
                            return;
                        }
                        // ページの取得(index番号は0が先頭のため1を引く)
                        using (PtlPage page = pages.get(pageToExtract - 1))
                        {
                            // 画像抽出
                            extractImageSimple(page, args[1], imageIndexNum);
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void extractImageSimple(PtlPage page, String imagefileName,
                                              int imageIndexNum)
        {
            // ページコンテント・画像エレメントの取得
            using (PtlContent content = page.getContent())
            using (PtlEditElements elems = content.getEditElements(PtlContent.GET_ELEMENT.GET_IMAGE))
            {
                int numElems = elems.getCount();
                if (numElems == 0)
                {
                    Console.WriteLine("指定されたページに画像エレメントがありませんでした。");
                    return; // 画像エレメントが無い場合はreturnする
                }
                //imageIndexNumのエラー処理
                if (numElems < imageIndexNum)
                {
                    Console.WriteLine("ERROR: image-index-numは対象ページの総画像数より" +
                                       "小さい値を指定してください。");
                    Console.WriteLine("総画像数：" + numElems);
                    printUsage();
                    return;
                }

                // OUTPUT_FORMATはFORMAT_AUTOで設定。
                PtlEditImage.OUTPUT_FORMAT format = PtlEditImage.OUTPUT_FORMAT.FORMAT_AUTO;
                // 画像エレメントの取得
                using (PtlEditElement elem = elems.get(imageIndexNum - 1))
                {
                    PtlEditElement.ELEMENT_TYPE type = elem.getType();
                    switch (type)
                    {
                        case PtlEditElement.ELEMENT_TYPE.TYPE_IMAGE:
                            PtlEditImage elemImage = (PtlEditImage)elem;
                            using (PtlParamOutput outputImage = new PtlParamOutput(imagefileName))
                            {
                                // 画像出力
                                elemImage.writeFile(outputImage, format);
                            }
                            break;
                        default:
                            Console.WriteLine("指定されたエレメントは画像ではありません。");
                            break;
                    }
                }
            }
        }
    }
}
