﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：指定したページから全テキスト抽出

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;
using System.Text;


namespace cookbook
{
    public class ExtractText
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: ExtractText in-pdf-file out-text-file" +
                               " page-to-extract");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の読み取り・判定
            // 出力PDFの名前はあとで渡すためにString型で保存する。
            String outputTextURI = args[1];
            int pageToExtract = int.Parse(args[2]);
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロード
                    doc.load(inputFile);
                    //ページ数を取得
                    int wholePageNum = doc.getPageCount();
                    //pageToExtractのエラー処理
                    if (wholePageNum < pageToExtract)
                    {
                        Console.WriteLine("ERROR: page-to-extractはPDFの総ページ数より" +
                                           "小さい値を指定してください。");
                        Console.WriteLine("総ページ数：" + wholePageNum);
                        printUsage();
                        return;
                    }

                    using (PtlPages pages = doc.getPages()) //ページコンテナの取得
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ERROR : ページコンテナが空");
                            return;
                        }

                        // ページの取得(パラメータindexは0が先頭のため1を引く)
                        using (PtlPage page = pages.get(pageToExtract - 1))
                        // ページコンテントの取得
                        using (PtlContent content = page.getContent())
                        // 文字抽出のパラメータクラス。今回は何も指定しない。
                        using (PtlParamExtractText plainParam = new PtlParamExtractText())
                        {
                            // 文字列抽出
                            String TextFromPdf = content.extractText(plainParam);

                            // 文字列の出力
                            outputTextFile(outputTextURI, TextFromPdf);
                            Console.WriteLine(TextFromPdf);
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }


        /**
         * テキストファイルを出力するための関数。 
         * 出力エンコードはシステムのデフォルト値を利用する。
         * 特に外部からの呼び出しを想定しないためprivateとする。
         * 
         * @param outputTextURI 出力ファイルのURI。
         * @param TextFromPdf 出力したいString型変数
         */
        private static void outputTextFile(String outputTextURI, String TextFromPdf)
        {
            Console.WriteLine("Output text URI :" + outputTextURI);
            try
            {
                using (StreamWriter sw = new StreamWriter(outputTextURI, false, Encoding.Default))
                {
                    sw.Write(TextFromPdf);
                }
            }
            catch (IOException e)
            {
                Console.WriteLine("IOException occured!!");
                Console.WriteLine(e.StackTrace);
            }
        }
    }
}
