﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：指定した矩形からテキスト抽出

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;
using System.Text;

namespace cookbook
{
    public class ExtractTextSetRect
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: ExtractTextSetRect in-pdf-file out-text-file" +
                               " page-to-extract overlap-ratio ignore-actual-text");
            Console.WriteLine("ignore-actual-text:");
            Console.WriteLine("0:ActualTextを無視しない 1:ActualTextを無視する");

        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 5)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の読み取り・判定
            // 出力PDFの名前はあとで渡すためにString型で保存する。
            String outputTextURI = args[1];
            int pageToExtract = int.Parse(args[2]);
            float overlapRatio = float.Parse(args[3]);
            bool ignoreActualText = true;
            try
            {
                ignoreActualText = readBoolArgs(args[4],
                                           "ignore-actual-textは1か0で指定してください。");

            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }

            try
        {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロード
                    doc.load(inputFile);
                    //ページ数を取得
                    int wholePageNum = doc.getPageCount();
                    //pageToExtractのエラー処理
                    if (wholePageNum < pageToExtract)
                    {
                        Console.WriteLine("ERROR: page-to-extractはPDFの総ページ数より" +
                                           "小さい値を指定してください。");
                        Console.WriteLine("総ページ数：" + wholePageNum);
                        printUsage();
                        return;
                    }

                    using (PtlPages pages = doc.getPages()) //ページコンテナの取得
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ERROR : ページコンテナが空");
                            return;
                        }

                        // ページの取得(パラメータindexは0が先頭のため1を引く)
                        using (PtlPage page = pages.get(pageToExtract - 1))
                        // ページコンテントの取得
                        using (PtlContent content = page.getContent())
                        // 文字抽出のパラメータクラス。
                        using (PtlParamExtractText paramExtractText = new PtlParamExtractText())
                        {
                            // テキスト抽出する矩形を指定
                            paramExtractText.appendRect(setRectCoordinate());
                            // テキストが矩形とどのくらい重なっていれば抽出するか設定
                            paramExtractText.setTextOverlapRatio(overlapRatio);
                            // ActialTextを無視するかどうか
                            paramExtractText.setIgnoreActualText(ignoreActualText);

                            // 文字列抽出
                            String TextFromPdf = content.extractText(paramExtractText);

                            // 文字列の出力
                            outputTextFile(outputTextURI, TextFromPdf);
                            Console.WriteLine(TextFromPdf);
                        }
                    }
                }
            }
        catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }


        /**
         * テキストファイルを出力するための関数。
         * 出力エンコードはシステムのデフォルト値を利用する。
         * 特に外部からの呼び出しを想定しないためprivateとする。
         *
         * @param outputTextURI 出力ファイルのURI。
         * @param TextFromPdf 出力したいString型変数
         */
        private static void outputTextFile(String outputTextURI, String TextFromPdf)
        {
            Console.WriteLine("Output text URI :" + outputTextURI);
            try
        {
                using (StreamWriter sw = new StreamWriter(outputTextURI, false, Encoding.Default))
                {
                    sw.Write(TextFromPdf);
                }
            }
        catch (IOException e)
            {
                Console.WriteLine("IOException occured!!");
                Console.WriteLine(e.StackTrace);
            }
        }

        /**
         * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         * 特に外部からの呼び出しを想定しないためprivateとする。
         *
         * @param br BufferedReader。数値の読み取りに使う。
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         */
        private static PtlRect setRectCoordinate()
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            PtlRect outputRect = new PtlRect();
            while (!isValueOkay)
            {
                Console.WriteLine("配置矩形の各数値を入力してください。");
                Console.Write("top (mm) : ");
                top = float.Parse(Console.ReadLine());
                Console.Write("bottom (mm) : ");
                bottom = float.Parse(Console.ReadLine());
                if (top < bottom)
                {
                    Console.WriteLine("topの値はbottomよりも大きい値を指定して" +
                                       "再度入力してください。");
                    continue;
                }
                Console.Write("left (mm) : ");
                left = float.Parse(Console.ReadLine());
                Console.Write("right (mm) : ");
                right = float.Parse(Console.ReadLine());
                if (right < left)
                {
                    Console.WriteLine("rightの値はleftよりも大きい値を指定して" +
                                       "再度入力してください。");
                    continue;
                }

                //矩形を正しく指定できた場合の処理
                isValueOkay = true;
                outputRect.setLeft(left);
                outputRect.setBottom(bottom);
                outputRect.setRight(right);
                outputRect.setTop(top);
            }
            return outputRect;
        }


        /**
         * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
         *
         * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
         * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
         * @return argsの数値を読み取った結果を戻す
         * @throws ArgumentException argsが0か1でなかった場合に発生。
         */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;

            // argsの読み取り
            switch (args)
            {
                case "0":
                    trueOrFalse = false;
                    break;
                case "1":
                    trueOrFalse = true;
                    break;
                default:
                    throw new ArgumentException(errorMessage);
            }

            return trueOrFalse;
        }
    }
}
