﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：カラー画像に最適化オプションを適用し、PDFを軽量化する。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class OptimizeImageColor
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: OptimizeImageColor in-pdf-file out-pdf-file" +
                               " 圧縮方法 画像品質");
            Console.WriteLine("圧縮方法：");
            Console.WriteLine("0:元のまま 1:お任せ 2:JPEG 3:JPEG2000 4:ZLIB");
            Console.WriteLine("画像品質：");
            Console.WriteLine("0:最高 1:高 2:中 3:低 4:最低");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                printUsage();
                return;
            }

            // コマンドライン引数の取得
            int compressType = int.Parse(args[2]);
            if ((compressType < 0) || (4 < compressType))
            {
                Console.WriteLine("圧縮方法 は0から4の範囲の整数値で指定してください。");
                printUsage();
                return;
            }
            int qualityType = int.Parse(args[3]);
            if ((qualityType < 0) || (4 < qualityType))
            {
                Console.WriteLine("画像品質 は0から4の範囲の整数値で指定してください。");
                printUsage();
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // 画像の最適化を行います。
                    optimizeDocWithParamImageColor(doc, compressType, qualityType);

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void optimizeDocWithParamImageColor(PtlPDFDocument doc, int compressType, int qualityType)
        {
            using (PtlParamOptimize paramOptimize = new PtlParamOptimize())
            // 画像最適化パラメーターの取得
            // カラー画像最適化パラメーターの取得
            using (PtlParamOptimizeImage paramOptimizeImage =
            paramOptimize.getParamOptimizeImage())
            using (PtlParamOptimizeImageColor paramOptimizeImageColor =
            paramOptimizeImage.getParamOptimizeImageColor())
            {

                // 圧縮方法を設定
                switch (compressType)
                {
                    case 0:
                        paramOptimizeImageColor.setCompress(PtlParamOptimizeImageColor.COMPRESS_TYPE.COMPRESS_SAME);
                        break;
                    case 1:
                        paramOptimizeImageColor.setCompress(PtlParamOptimizeImageColor.COMPRESS_TYPE.COMPRESS_AUTO);
                        break;
                    case 2:
                        paramOptimizeImageColor.setCompress(PtlParamOptimizeImageColor.COMPRESS_TYPE.COMPRESS_JPEG);
                        break;
                    case 3:
                        paramOptimizeImageColor.setCompress(PtlParamOptimizeImageColor.COMPRESS_TYPE.COMPRESS_JPEG2K);
                        break;
                    case 4:
                        paramOptimizeImageColor.setCompress(PtlParamOptimizeImageColor.COMPRESS_TYPE.COMPRESS_ZLIB);
                        break;
                }
                // 画像品質を設定
                switch (qualityType)
                {
                    case 0:
                        paramOptimizeImageColor.setQuality(PtlParamOptimizeImageColor.QUALITY_TYPE.QUALITY_MAX);
                        break;
                    case 1:
                        paramOptimizeImageColor.setQuality(PtlParamOptimizeImageColor.QUALITY_TYPE.QUALITY_HIGH);
                        break;
                    case 2:
                        paramOptimizeImageColor.setQuality(PtlParamOptimizeImageColor.QUALITY_TYPE.QUALITY_MIDDLE);
                        break;
                    case 3:
                        paramOptimizeImageColor.setQuality(PtlParamOptimizeImageColor.QUALITY_TYPE.QUALITY_LOW);
                        break;
                    case 4:
                        paramOptimizeImageColor.setQuality(PtlParamOptimizeImageColor.QUALITY_TYPE.QUALITY_MIN);
                        break;
                }

                // 最適化する画像の対象Filterを設定
                paramOptimizeImage.setValidFilter(PtlParamOptimizeImage.FILTER_TYPE.FILTER_ALL & ~PtlParamOptimizeImage.FILTER_TYPE.FILTER_JBIG2Decode);
                // 画像のダウンサンプリングを行う最低サンプル数を設定
                paramOptimizeImage.setMinSampleSize(110);

                paramOptimizeImageColor.setSourcePPI(150);
                paramOptimizeImageColor.setTargetPPI(75);
                paramOptimizeImageColor.setDownSampling(PtlParamOptimizeImageDownSampling.DOWNSAMPLING_TYPE.DOWNSAMPLING_BICUBIC);
                paramOptimizeImageColor.setMinDownSamplingRate(0.65f);


                // 最適化の実行
                doc.optimize(paramOptimize);
            }
        }
    }
}
