﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：カラー画像に最適化オプションを適用し、PDFを軽量化します。
        最適化の際に画像ダウンサンプリングのタイプを選択します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class OptimizeImageSetDownSamplingType
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: OptimizeImageSetDownSampling in-pdf-file" +
                               " out-pdf-file downsampling-type");
            Console.WriteLine("downsampling-type：");
            Console.WriteLine("0:ダウンサンプリングしない 1:バイキュービック法" +
                               " 2:バイリニア法 3:ニアレストネイバー法");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                printUsage();
                return;
            }

            // コマンドライン引数の取得
            int downSamplingType = int.Parse(args[2]);
            if ((downSamplingType < 0) || (3 < downSamplingType))
            {
                Console.WriteLine("downsampling-type は0から3の範囲の整数値で指定してください。");
                printUsage();
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // 画像の最適化を行います。
                    optimizeDocSetDownSamplingType(doc, downSamplingType);

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void optimizeDocSetDownSamplingType(PtlPDFDocument doc,
                                                          int downSamplingType)
        {
            using (PtlParamOptimize paramOptimize = new PtlParamOptimize())
            // 画像最適化パラメーターの取得
            // カラー画像最適化パラメーターの取得
            using (PtlParamOptimizeImage paramOptimizeImage =
            paramOptimize.getParamOptimizeImage())
            using (PtlParamOptimizeImageColor paramOptimizeImageColor =
            paramOptimizeImage.getParamOptimizeImageColor())
            {

                // ダウンサンプリングのタイプを設定
                switch (downSamplingType)
                {
                    case 0:
                        paramOptimizeImageColor.setDownSampling(PtlParamOptimizeImageDownSampling.DOWNSAMPLING_TYPE.DOWNSAMPLING_NONE);
                        break;
                    case 1:
                        paramOptimizeImageColor.setDownSampling(PtlParamOptimizeImageDownSampling.DOWNSAMPLING_TYPE.DOWNSAMPLING_BICUBIC);
                        break;
                    case 2:
                        paramOptimizeImageColor.setDownSampling(PtlParamOptimizeImageDownSampling.DOWNSAMPLING_TYPE.DOWNSAMPLING_AVERAGE);
                        break;
                    case 3:
                        paramOptimizeImageColor.setDownSampling(PtlParamOptimizeImageDownSampling.DOWNSAMPLING_TYPE.DOWNSAMPLING_SUBSAMPLING);
                        break;
                }

                // 最適化する画像の対象Filterを設定
                paramOptimizeImage.setValidFilter(PtlParamOptimizeImage.FILTER_TYPE.FILTER_ALL & ~PtlParamOptimizeImage.FILTER_TYPE.FILTER_JBIG2Decode);
                // 画像のダウンサンプリングを行う最低サンプル数を設定
                paramOptimizeImage.setMinSampleSize(110);

                // その他のダウンサンプリングの値を設定
                paramOptimizeImageColor.setSourcePPI(150);
                paramOptimizeImageColor.setTargetPPI(75);
                paramOptimizeImageColor.setMinDownSamplingRate(0.65f);
                paramOptimizeImageColor.setCompress(PtlParamOptimizeImageColor.COMPRESS_TYPE.COMPRESS_AUTO);
                paramOptimizeImageColor.setQuality(PtlParamOptimizeImageColor.QUALITY_TYPE.QUALITY_MIDDLE);


                // 最適化の実行
                doc.optimize(paramOptimize);
            }
        }
    }
}
